/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization.dto;

import static java.util.Objects.isNull;
import static java.util.Optional.ofNullable;
import static java.util.stream.Stream.of;

import org.mule.runtime.extension.api.error.ErrorMapping;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

public class ParameterValueContainer {

  private Object object = null;
  private Boolean aBoolean = null;
  private String aString = null;
  private Character character = null;
  private Byte aByte = null;
  private Short aShort = null;
  private Integer anInteger = null;
  private Long aLong = null;
  private Float aFloat = null;
  private Double aDouble = null;
  private String expression = null;
  private ComponentAstDTO singlePojoComponentAstDto = null;
  private List<ComponentAstDTO> listOfPojoComponentAstDto = null;
  private List<ErrorMapping> listOfErrorMappings = null;

  public ParameterValueContainer(String expression, Object fixedValue) {
    if (of(expression, fixedValue).filter(obj -> !isNull(obj)).count() > 1) {
      throw new IllegalArgumentException("Only one parameter can be not null");
    }

    if (expression != null) {
      this.expression = expression;
      return;
    }

    if (fixedValue instanceof Boolean) {
      this.aBoolean = (Boolean) fixedValue;
      return;
    }

    if (fixedValue instanceof String) {
      this.aString = (String) fixedValue;
      return;
    }

    if (fixedValue instanceof Character) {
      this.character = (Character) fixedValue;
      return;
    }

    if (fixedValue instanceof Byte) {
      this.aByte = (Byte) fixedValue;
      return;
    }

    if (fixedValue instanceof Short) {
      this.aShort = (Short) fixedValue;
      return;
    }

    if (fixedValue instanceof Integer) {
      this.anInteger = (Integer) fixedValue;
      return;
    }

    if (fixedValue instanceof Long) {
      this.aLong = (Long) fixedValue;
      return;
    }

    if (fixedValue instanceof Float) {
      this.aFloat = (Float) fixedValue;
      return;
    }

    if (fixedValue instanceof Double) {
      this.aDouble = (Double) fixedValue;
      return;
    }

    if (fixedValue instanceof ComponentAstDTO) {
      this.singlePojoComponentAstDto = (ComponentAstDTO) fixedValue;
      return;
    }

    if (isAListOfComponentAstDTOs(fixedValue)) {
      this.listOfPojoComponentAstDto = (List<ComponentAstDTO>) fixedValue;
      return;
    }

    if (isAListOfErrorMappings(fixedValue)) {
      this.listOfErrorMappings = (List<ErrorMapping>) fixedValue;
      return;
    }

    this.object = fixedValue;
  }

  public Object getContainedValue() {
    // At most one item will be non-null
    return of(object, aBoolean, aString, character, aByte, aShort, anInteger, aLong, aFloat, aDouble, expression,
              singlePojoComponentAstDto,
              listOfPojoComponentAstDto,
              listOfErrorMappings)
                  .filter(Objects::nonNull).findFirst().orElse(null);
  }

  @Override
  public String toString() {
    if (object != null) {
      return "ParameterValueContainer{" + "object=" + object + '}';
    }

    if (aBoolean != null) {
      return "ParameterValueContainer{" + "aBoolean=" + aBoolean + '}';
    }

    if (aString != null) {
      return "ParameterValueContainer{" + "aString=" + aString + '}';
    }

    if (character != null) {
      return "ParameterValueContainer{" + "character=" + character + '}';
    }

    if (aByte != null) {
      return "ParameterValueContainer{" + "aByte=" + aByte + '}';
    }

    if (aShort != null) {
      return "ParameterValueContainer{" + "aShort=" + aShort + '}';
    }

    if (anInteger != null) {
      return "ParameterValueContainer{" + "anInteger=" + anInteger + '}';
    }

    if (aLong != null) {
      return "ParameterValueContainer{" + "aLong=" + aLong + '}';
    }

    if (aFloat != null) {
      return "ParameterValueContainer{" + "aFloat=" + aFloat + '}';
    }

    if (aDouble != null) {
      return "ParameterValueContainer{" + "aDouble=" + aDouble + '}';
    }

    if (expression != null) {
      return "ParameterValueContainer{" + "expression=" + expression + '}';
    }

    if (singlePojoComponentAstDto != null) {
      return "ParameterValueContainer{" + "singlePojoComponentAstDto=" + singlePojoComponentAstDto + '}';
    }

    if (listOfPojoComponentAstDto != null) {
      return "ParameterValueContainer{" + "listOfPojoComponentAstDto=" + listOfPojoComponentAstDto + '}';
    }

    if (listOfErrorMappings != null) {
      return "ParameterValueContainer{" + "listOfErrorMappings=" + listOfErrorMappings + '}';
    }

    return "ParameterValueContainer{}";

  }

  public boolean isExpression() {
    return expression != null;
  }

  /**
   * @return true if it holds a value for the right type, false otherwise
   */
  public boolean isFixedValue() {
    return expression == null;
  }

  /**
   * @return the left value
   */
  public String getExpression() {
    return expression;
  }

  /**
   * @return the right value
   */
  public Object getFixedValue() {
    // At most one item will be non-null
    return of(object, aBoolean, aString, character, aByte, aShort, anInteger, aLong, aFloat, aDouble, singlePojoComponentAstDto,
              listOfPojoComponentAstDto,
              listOfErrorMappings)
                  .filter(Objects::nonNull).findFirst().orElse(null);
  }

  public Optional<Object> getValue() {
    return ofNullable(this.getContainedValue());
  }

  private boolean isAListOfComponentAstDTOs(Object right) {
    return (right instanceof List && !((List) right).isEmpty() && ((List) right).get(0) instanceof ComponentAstDTO);
  }

  private boolean isAListOfErrorMappings(Object right) {
    return (right instanceof List && !((List) right).isEmpty() && ((List) right).get(0) instanceof ErrorMapping);
  }

}
