/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.serialization;

import static java.lang.System.lineSeparator;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.requireNonNull;
import static java.util.regex.Pattern.compile;
import static org.apache.commons.io.IOUtils.toInputStream;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.SequenceInputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.input.BoundedInputStream;

/**
 * This class deals with the adding/reading into/from an input stream the metadata associated to
 * {@link org.mule.runtime.ast.api.ArtifactAst} serialization. This metadata is generated during the artifact's serialization and
 * must be added to the resulting {@link InputStream} so that it can be then read in order to use the correct deserializing
 * implementation when deserializing.
 * <p>
 * The metadata is added to the stream in the form of a header string in the first line of the {@link InputStream}. This header
 * string is composed by the serializer's id and the version, delimited by a pound sign (#). The max allowed length of the header
 * string is {@value MAX_HEADER_LINE_LENGTH} as determined by the {@link #MAX_HEADER_LINE_LENGTH} constant.
 */
public class ArtifactAstSerializerMetadataSerializer {

  // Max header size + possible /r/n
  private static final long MAX_HEADER_LINE_LENGTH = 1002;

  public static final String DELIMITER = "#";
  public static final String HEADER_REGEX = "^([^#]+)[#]([^#]+)[#]$";
  private final Pattern headerRegexPattern = compile(HEADER_REGEX);

  /**
   * This method reads the metadata from an {@link InputStream} by reading the first line which should be a header string in a
   * serialized {@link org.mule.runtime.ast.api.ArtifactAst}). This header string is composed by the serializer's id and the
   * version, delimited by a pound sign (#).
   * <p>
   * The max allowed length of the header string is {@value MAX_HEADER_LINE_LENGTH} as determined by the
   * {@link #MAX_HEADER_LINE_LENGTH} constant. Since this method only read the first line, the rest of the input stream will be
   * still available for consumption.
   *
   * @param inputStream a serialized {@link org.mule.runtime.ast.api.ArtifactAst} generated by an
   *                    {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer}
   * @return a {@link ArtifactAstSerializerMetadata} instance containing the id and version indicated in the input stream's
   *         metadata.
   * @throws IOException If an I/O error occurs
   */
  public ArtifactAstSerializerMetadata readArtifactAstSerializerMetadataFromInputStream(InputStream inputStream)
      throws IOException {
    // Using a bounded input stream to prevent a DDos attack
    BoundedInputStream boundedInputStream = new BoundedInputStream(inputStream, MAX_HEADER_LINE_LENGTH);
    BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(boundedInputStream));
    String headerString = bufferedReader.readLine();

    // The expected format is: ID#VERSION# -> Beginning + (a string with no # followed by a #) * twice
    // I could just as well validate the format of a serializer id or a version. TBD
    Matcher matcher = headerRegexPattern.matcher(headerString);

    if (!matcher.matches()) {
      throw new IllegalArgumentException("The serialized artifact input stream has an invalid header.");
    }

    return new ArtifactAstSerializerMetadata(matcher.group(1), matcher.group(2));
  }

  /**
   * This method serializes the {@link ArtifactAstSerializerMetadata} into a header string and adds it as a first line to the
   * provided serialized {@link org.mule.runtime.ast.api.ArtifactAst} which should be an {@link InputStream}
   * 
   * @param inputStream                   the serialized {@link org.mule.runtime.ast.api.ArtifactAst} created with an
   *                                      {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer}
   * @param artifactAstSerializerMetadata the {@link ArtifactAstSerializerMetadata} that represents the information associated
   *                                      with the {@link org.mule.runtime.ast.api.serialization.ArtifactAstSerializer} used to
   *                                      generate the given {@link InputStream}
   * @return an {@link InputStream} whose first line is the serialized {@link ArtifactAstSerializerMetadata} and is followed by
   *         the contents of the given serialized {@link org.mule.runtime.ast.api.ArtifactAst}'s {@link InputStream}
   */
  public InputStream addArtifactAstSerializerMetadataToInputStream(InputStream inputStream,
                                                                   ArtifactAstSerializerMetadata artifactAstSerializerMetadata) {
    requireNonNull(inputStream, "inputStream");
    requireNonNull(artifactAstSerializerMetadata, "artifactAstSerializerMetadata");

    String headerString = artifactAstSerializerMetadata.getSerializerId() + DELIMITER
        + artifactAstSerializerMetadata.getSerializerVersion() + DELIMITER
        + lineSeparator();
    InputStream headerInputStream = toInputStream(headerString, UTF_8);
    return new SequenceInputStream(headerInputStream, inputStream);
  }
}
