/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.graph.internal.cycle;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;

import org.jgrapht.Graph;
import org.jgrapht.alg.cycle.CycleDetector;
import org.jgrapht.graph.SimpleDirectedGraph;
import org.slf4j.Logger;

/**
 * Given a {@link Graph} it will use {@link CycleDetector} to find and collect cycles and generate a new {@link Graph}
 * without cycle edges. The comparator is used to sort the order of the cycles when removing them.
 *
 * @param <V>
 * @param <T>
 */
public class GraphCycleRemover<V, T> {

  private static final Logger LOGGER = getLogger(GraphCycleRemover.class);

  private final Graph<V, T> originalGraph;
  private Comparator<V> cycleComparator;

  public GraphCycleRemover(Graph<V, T> originalGraph, Comparator<V> cycleComparator) {
    this.cycleComparator = cycleComparator;
    this.originalGraph = originalGraph;
  }

  public Graph<V, T> removeCycles() {
    SimpleDirectedGraph<V, T> newGraph =
        new SimpleDirectedGraph<>(originalGraph.getVertexSupplier(), originalGraph.getEdgeSupplier(),
                                  originalGraph.getType().isWeighted());
    // Add all vertex
    originalGraph.vertexSet().forEach(newGraph::addVertex);
    // Add edges from original originalGraph
    originalGraph.edgeSet()
        .forEach(edge -> newGraph.addEdge(originalGraph.getEdgeSource(edge), originalGraph.getEdgeTarget(edge)));

    CycleDetector<V, T> cycleDetector = new CycleDetector<>(newGraph);
    while (cycleDetector.detectCycles()) {
      // Loop through vertices trying to find disjoint cycles.
      List<V> cycles = new ArrayList<>(cycleDetector.findCycles());
      cycles.sort(cycleComparator);

      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Cycles detected {}", cycles);
      }
      for (V toVertex : cycles) {
        // Get all vertices involved with this vertex.
        Optional<V> fromVertexOptional = cycleDetector.findCyclesContainingVertex(toVertex).stream()
            .filter(vertex -> newGraph.getEdge(vertex, toVertex) != null)
            .findAny();

        if (fromVertexOptional.isPresent()) {
          T edgeRemoved = newGraph.removeEdge(fromVertexOptional.get(), toVertex);
          if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Removing cycle edge {}", edgeRemoved);
          }
        }
      }
      cycleDetector = new CycleDetector<>(newGraph);
    }

    return newGraph;
  }

}
