/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.internal.util.collection;

import static java.util.Collections.singleton;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;

/**
 * {@link Collector} that returns an immutable {@link Set}
 *
 * @param <T> the generic type of the elements in the {@link Set}
 * @since 1.0
 */
public class ImmutableSetCollector<T> implements Collector<T, Set<T>, Set<T>> {

  @Override
  public Supplier<Set<T>> supplier() {
    return HashSet::new;
  }

  @Override
  public BiConsumer<Set<T>, T> accumulator() {
    return Set::add;
  }

  @Override
  public BinaryOperator<Set<T>> combiner() {
    return (left, right) -> {
      left.addAll(right);
      return left;
    };
  }

  @Override
  public Function<Set<T>, Set<T>> finisher() {
    return Collections::unmodifiableSet;
  }

  @Override
  public Set<Characteristics> characteristics() {
    return singleton(Characteristics.UNORDERED);
  }
}
