 /**
* This module provides functionality used for the tokenization service
**/
%dw 2.0

import createSequencer, nextSequence from org::mule::tokenization::Sequencer

type Sequencer = Object

type TokenizationExpression = { expression: String, format: String }

type ExpressionTree = { name: String, format?: String, children: { _?: ExpressionTree } }

type TokenizationMatch = { format: String, data: Any }

/**
* Helper function to create an ExpressionTree based on o set of selection expression.
* This tree is used in the `collect` and in `substitute` to only traverse the parts of the data that is required.
* 
* === Example
* 
* This example shows how it merges all the expressions into a single tree
* 
* ==== Source
* 
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* toExpressionTree([{expression: "#[payload.a.b]", format: "ssn"},{expression: "#[payload.a.c], format: "ssn"}])
* ----
* 
* ==== Output
* 
* [source,JSON,linenums]
* ----
* {
*    "name": "payload",
*    "children": {
*      "a": {
*        "name": "a",
*        "children": {
*          "b": {
*            "name": "b",
*            "format": "ssn",
*            "children": {}
*          },
*          "c": {
*            "name": "c",
*            "format": "ssn",
*            "children": {}
*          }
*        }
*      }
*    }
*  }
* ----
**/
fun toExpressionTree(expression: Array<TokenizationExpression>) = do {


  fun merge(@DesignOnlyType() expression: ExpressionTree, newExpressionPart: Array<String>, format: String): ExpressionTree = do {
    newExpressionPart match {
      case [] -> expression
      case [x ~ xs] -> do {
        {
          name: expression.name!,
          (format: expression.format!) if (expression.format?),
          children: 
            // We replace the new child if it exists
            // Or we create a new expression if it does not
            if (expression.children[x]?)
              (expression.children - x) ++ do {
                var expressionTree = merge(expression.children[x]!, xs, format)
                ---
                {
                  (expressionTree.name) : expressionTree
                }
              }
            else
              expression.children ++ do {
                var expressionTree = createNewExpressionTree(newExpressionPart, format)
                ---
                {
                  (expressionTree.name) : expressionTree
                }
              }
        }
      }
    }
  }

  fun createNewExpressionTree(newExpressionPart: Array<String>, format: String): ExpressionTree = do {
    newExpressionPart match {
      case [x ~ xs] -> {
        name: x,
        (format: format) if isEmpty(xs), // Only if it is a leaf
        children: 
          if (isEmpty(xs)) {} else do {
            var child = createNewExpressionTree(xs, format)
            ---
            {
              (child.name) : child
            }
          }
      }
    }
  }

  fun getExpressionParts(expression: String): Array<String> = expression[1 to -2] splitBy "."

  var initialExpression = createNewExpressionTree(getExpressionParts(expression[0].expression), expression[0].format)
  ---
  expression[1 to -1] default [] reduce ((expression, accumulator = initialExpression) -> do {
      var partsWithoutTheRoot = (getExpressionParts(expression.expression))[1 to -1] default []
      ---
      merge(accumulator, partsWithoutTheRoot, expression.format)
    })
}

/**
* Replace the matching expressions with the substitution values. The values are going to be in the order that were matched in the collect
* 
* === Example
* 
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from Tokenization
* output application/json
* ---
* substitute(payload, ["SSN-1", "CC1", "CC2","CC3"], [ { format: "ssn", expression: "#[payload.ssn]" }, { format: "ccn",expression: "#[payload.credit_cards.ccn]" } ])
* ----
* 
* ==== Input
* 
* [source,json, linenums]
* ----
* {
*    "name" : "Richard Fort",
*    "address" : "Miamee 999",
*    "ssn" : "1122334455",
*    "credit_cards" : [
*        {
*            "bank" : "HSBC",
*            "vendor" : "VISA",
*            "ccn": "1234-5678-9012-3456"
*        },
*        {
*            "bank" : "Galicia",
*            "vendor" : "Amex",
*            "ccn": "0987-6543-2109-8765"
*        },
*        {
*            "bank" : "Galicia",
*            "vendor" : "VISA",
*            "ccn": "1357-2468-9753-0864"
*        }
*    ]
* }
* ----
* 
* ==== Output
* 
* [source,json, linenums]
* ----
* {
*  "name": "Richard Fort",
*  "address": "Miamee 999",
*  "ssn": "SSN-1",
*  "credit_cards": [
*    {
*      "bank": "HSBC",
*      "vendor": "VISA",
*      "ccn": "CC1"
*    },
*    {
*      "bank": "Galicia",
*      "vendor": "Amex",
*      "ccn": "CC2"
*    },
*    {
*      "bank": "Galicia",
*      "vendor": "VISA",
*      "ccn": "CC3"
*    }
*  ]
* }
* ----
**/
fun substitute(value: Any, substitutions: Array<Any>, expressions: Array<TokenizationExpression>): Any = do {

  var expressionTree = toExpressionTree(expressions)

  fun doSubstitute(value: Any, substitutions: Array<Any>, @DesignOnlyType() expression: ExpressionTree, sequencer: Sequencer, insideAttribute: Boolean) = do {
    value match {
      case o is Object -> do {
        o mapObject ((value, key, index) -> do {
            var keyName = if(insideAttribute) "@" ++ key as String else key as String
            var childNode = expression.children[keyName]
            ---
            if (childNode != null)
                  if (isEmpty(childNode.children)) //Is Leaf then substitute
                    (key): substitutions[nextSequence(sequencer)]
                  else if(key.@?)
                    (key) @((doSubstitute(key.@, substitutions, expression.children[keyName]!,sequencer, true))): doSubstitute(value, substitutions, expression.children[keyName]!, sequencer, insideAttribute)
                  else
                    (key): doSubstitute(value, substitutions, expression.children[keyName]!, sequencer, insideAttribute)
            else
              {
                (key) : value
              }
          })
      }
      case a is Array -> a map ((item, index) -> doSubstitute(item, substitutions, expression, sequencer, insideAttribute))
      else -> if(isEmpty(expression.children) and not (expression.name startsWith "@"))
                substitutions[nextSequence(sequencer)]
              else
                value
    }
  }
  ---
  doSubstitute(value, substitutions, expressionTree, createSequencer(), false)
}


/**
* Collects all elements that need to be tokenized according to the `TokenizationExpression` expressions
*
* === Example
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from Tokenization
* output application/json
* ---
* collect(payload, [ { format: "ssn", expression: "payload.ssn" }, { format: "ccn",expression: "payload.credit_cards.ccn" } ])
* ----
*
* ==== Input
* [source,json, linenums]
* ----
* {
*    "name" : "Richard Fort",
*    "address" : "Miamee 999",
*    "ssn" : "1122334455",
*    "credit_cards" : [
*        {
*            "bank" : "HSBC",
*            "vendor" : "VISA",
*            "ccn": "1234-5678-9012-3456"
*        },
*        {
*            "bank" : "Galicia",
*            "vendor" : "Amex",
*            "ccn": "0987-6543-2109-8765"
*        },
*        {
*            "bank" : "Galicia",
*            "vendor" : "VISA",
*            "ccn": "1357-2468-9753-0864"
*        }
*    ]
* }
* ----
* ==== Output
*
* [source,json, linenums]
* ----
*[
*   {
*     "format": "ssn",
*     "value": "1122334455"
*   },
*   {
*     "format": "ccn",
*     "value": "1234-5678-9012-3456"
*   },
*   {
*     "format": "ccn",
*     "value": "0987-6543-2109-8765"
*   },
*   {
*     "format": "ccn",
*     "value": "1357-2468-9753-0864"
*   }
* ]
* ----
*/
fun collect(value: Any, tokenizationExpressions: Array<TokenizationExpression>): Array<TokenizationMatch> = do {
  var tokenizationExprByExpression = toExpressionTree(tokenizationExpressions)

  fun doCollect(value: Any, @DesignOnlyType() expression: ExpressionTree, insideAttribute: Boolean): Array<TokenizationMatch> = do {
    value match {
      case o is Object -> do {
        flatten(o pluck ((value, key, index) -> do {
            var keyName = if(insideAttribute) "@" ++ key as String else key as String
            var childNode = expression.children[keyName]
            ---
            if (childNode != null)
              if (isEmpty(childNode.children)) // is Leaf then we need to collect it
                [
                  {
                    format: childNode.format default "",
                    data: value
                  }
                ]
              else if(key.@?)
                doCollect(value, expression.children[keyName]!, insideAttribute) ++ doCollect(key.@, expression.children[keyName]!, true)
              else
                doCollect(value, expression.children[keyName]!, insideAttribute)
            else
              []
          }))
      }
      case a is Array -> a flatMap ((item, index) -> doCollect(item, expression, insideAttribute))
      else ->
            if(isEmpty(expression.children) and not (expression.name startsWith "@"))
              [
                {
                  format: expression.format default "",
                  data: value
                }
              ]
            else
              []
    }
  }
  ---
  if(value == null)
    []
  else
    doCollect(value, tokenizationExprByExpression, false)
}

