%dw 2.7
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Executor
import fail from dw::Runtime

/**
* Transforms an operation by applying a function that adapts its input type
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to transform
* | `mapper` | &#40;NewOpParam&#41; &#45;&#62; OpParam | A function that adapts its input to a more useful one
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* var numbers: Operation<Array<Number>, Array<Number>, Nothing, {}> = {
*   name: "numbers",
*   displayName: "Gets you a bunch of numbers",
*   executor: (params, connectionInstance) ->
*     success((params[0] to (params[0] + params[1])) as Array<Number>)
* }
*
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* var betterNumbers = numbers
*   mapInputOperation (params: NumbersParam) -> [params.from, params.count]
* ----
*/
fun mapInputOperation<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpParam>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    mapper: (NewOpParam) -> OpParam)
    : Operation<NewOpParam, OpResultType, OpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update { case executor at .executor -> mapInput(operation.executor, mapper)}

/**
* Transforms a paginated operation by applying a function that adapts its input type
*/
fun mapInputPaginatedOperation<OpParam, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpParam, ItemType, NextOpParam>(
    operation: PaginatedOperation<OpParam, NextOpParam, Page<ItemType, NextOpParam>, OpResultErrorType, OpConnectionType>,
    mapper: (NewOpParam) -> OpParam)
    : PaginatedOperation<NewOpParam, NextOpParam, Page<ItemType, NextOpParam>, OpResultErrorType, OpConnectionType> =
      mapInputOperation(operation, mapper) as PaginatedOperation<NewOpParam, NextOpParam, Page<ItemType, NextOpParam>, OpResultErrorType, OpConnectionType>

/**
* Transforms an operation by applying a function on its successful and failed results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to transform
* | `successMapper` | &#40;OpResultType&#41; &#45;&#62; NewOpResultType | A function that receives the successful result and improves it somehow
* | `failureMapper` | &#40;OpResultErrorType&#41; &#45;&#62; NewOpResultErrorType | A function that receives the error and transforms it to a _better_ error
* |===
*
*/
fun mapOutputOperation<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultType, NewOpResultErrorType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    successMapper: (OpResultType) -> NewOpResultType,
    failureMapper: (OpResultErrorType) -> NewOpResultErrorType
    ): Operation<OpParam, NewOpResultType, NewOpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update { case executor at .executor -> mapResult(operation.executor, successMapper, failureMapper)}

/**
* Transforms an operation by applying a function on its successful results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to transform
* | `mapper` | &#40;OpResultType&#41; &#45;&#62; NewOpResultType | A function that receives the successful result and improves it somehow
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* type NumbersResult = {
*   numbers: Array<Number>
* }
*
* var numbers: Operation<NumbersParam, NumbersResult, Nothing, {}> = {
*   name: "numbers",
*   displayName: "Gets you a bunch of numbers",
*   executor: (params, connectionInstance) ->
*     success({
*       numbers: (params.from to (params.from + params.count)) as Array<Number>
*     })
* }
*
* var simplerNumbers = numbers
*   mapOutputSuccessOperation (data) -> data.numbers
* ----
*/
fun mapOutputSuccessOperation<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    mapper: (OpResultType) -> NewOpResultType)
    : Operation<OpParam, NewOpResultType, OpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update { case executor at .executor -> mapSuccessResult(operation.executor, mapper)}

/**
* Transforms an operation by applying a function on its failed results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to transform
* | `mapper` | &#40;OpResultErrorType&#41; &#45;&#62; NewOpResultErrorType | A function that receives the error and transforms it to a _better_ error
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* var numbers: Operation<NumbersParam, Array<Number>, Number, {}> = {
*   name: "numbers",
*   displayName: "Gets you a bunch of numbers",
*   executor: (params, connectionInstance) ->
*     if (params.from > 100)
*       failure(params.from, "Number too big")
*     else
*       success((params.from to (params.from + params.count)) as Array<Number>)
* }
*
* var betterNumbers = numbers
*   mapOutputFailureOperation (error) -> { message: "Failed because the number was too big", failedFrom: error }
* ----
*/
fun mapOutputFailureOperation<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultErrorType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    mapper: (OpResultErrorType) -> NewOpResultErrorType)
    : Operation<OpParam, OpResultType, NewOpResultErrorType, OpConnectionType> = do {
    var executor = operation.executor
    ---
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update {
            case executor at .executor -> mapFailureResult(operation.executor, mapper)
        }
}

/**
* Transforms a paginated operation by applying a function on its failed results
*/
fun mapOutputFailurePaginatedOperation<OpParam, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultErrorType, ItemType, NextOpParam>(
    operation: PaginatedOperation<OpParam, NextOpParam, Page<ItemType, NextOpParam>, OpResultErrorType, OpConnectionType>,
    mapper: (OpResultErrorType) -> NewOpResultErrorType)
    : PaginatedOperation<OpParam, NextOpParam, Page<ItemType, NextOpParam>, NewOpResultErrorType, OpConnectionType> =
      mapOutputFailureOperation(operation, mapper) as PaginatedOperation<OpParam, NextOpParam, Page<ItemType, NextOpParam>, NewOpResultErrorType, OpConnectionType>


/**
* Decorates a given operation by running an additional executor after
* successful executions. This new executor has access to the original
* connection and returns a `Result<T, E>` value that will become the result
* of the decorated operation.
*
* This combinator is useful for performing additional checks on the result (a
* 200 OK example may still be failure depending on the usecase) and creating
* operations that chain multiple HTTP requests.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; |  The operation to decorate
* | `executor` | &#40;OpResultType, OpConnectionType&#41; &#45;&#62; Result<NewOpResultType, NewOpResultErrorType&#62; | The executor to run on successful execution of the decorated operation
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.8
*
* type AddParam = {
*   left: Number,
*   right: Number
* }
* type AddFailure = {
*   arg: Number,
*   minValid: Number,
*   maxValid: Number
* }
*
* var add: Operation<{ left: Number, right: Number }, Number, ResultFailure<AddFailure, Error>, Null> = {
*   name: "Add",
*   displayName: "Adds two numbers (result = left + right)",
*   executor: (parameter, requestBuilder) -> do {
*     var minValid = 0
*     var maxValid = 100
*     ---
*     if (maxValid < parameter.left or parameter.left < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`left` was outside the expected range")
*     else if (maxValid < parameter.right or parameter.right < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`right` was outside the expected range")
*     else
*       success(parameter.left + parameter.right)
*   }
* }
*
* var failsForBigResults = add
*   onSuccessRunExecutor do {
*     var maxResult = 100
*     ---
*     if (maxResult < $) 
*       failure({
*         result: $,
*         maxResult: maxResult
*       }, "Result was too big!")
*     else
*       success($)
*   }
* ----
*
*/
fun onSuccessRunExecutor<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultType, NewOpResultErrorType <: ResultFailure>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    executor: (OpResultType, OpConnectionType) -> Result<NewOpResultType, NewOpResultErrorType>
): Operation<OpParam, NewOpResultType, OpResultErrorType | NewOpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update {
            case operationExecutor at .executor ->
                (inputType: OpParam, connectionExecutor: OpConnectionType): Result<NewOpResultType, OpResultErrorType | NewOpResultErrorType> ->
                    operationExecutor(inputType, connectionExecutor) match {
                        case result is ResultSuccess<OpResultType> ->
                            executor(result.value, connectionExecutor)
                        case is OpResultErrorType -> $
                   }
        }

/**
* Decorates a given operation by running an additional executor after
* failed executions. This new executor has access to the original
* connection and returns a `Result<T, E>` value that will become the result
* of the decorated operation.
*
* This combinator is useful for performing error recovery, either by
* transforming an error result into a useful one or by performing a fallback
* HTTP request.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to decorate
* | `executor` | &#40;OpResultErrorType, OpConnectionType&#41; &#45;&#62; Result<NewOpResultType, NewOpResultErrorType&#62; | The executor to run on failed executions of the decorated operation
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.8
*
* type AddParam = {
*   left: Number,
*   right: Number
* }
* type AddFailure = {
*   arg: Number,
*   minValid: Number,
*   maxValid: Number
* }
*
* var add: Operation<{ left: Number, right: Number }, Number, ResultFailure<AddFailure, Error>, Null> = {
*   name: "Add",
*   displayName: "Adds two numbers (result = left + right)",
*   executor: (parameter, requestBuilder) -> do {
*     var minValid = 0
*     var maxValid = 100
*     ---
*     if (maxValid < parameter.left or parameter.left < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`left` was outside the expected range")
*     else if (maxValid < parameter.right or parameter.right < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`right` was outside the expected range")
*     else
*       success(parameter.left + parameter.right)
*   }
* }
*
* var succeedsWithErrorCode = add
*   onFailureRunExecutor
*     if ($.error.value.arg < $.error.value.minValid) 
*       success(-1) // Argument was too small
*     else if ($.error.value.maxValid < $.error.value.arg) 
*       success(-2) // Argument was too big
*     else
*       $
* ----
*
*/
fun onFailureRunExecutor<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultType, NewOpResultErrorType <: ResultFailure>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    executor: (OpResultErrorType, OpConnectionType) -> Result<NewOpResultType, NewOpResultErrorType>
): Operation<OpParam, OpResultType | NewOpResultType, NewOpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update {
            case operationExecutor at .executor ->
                (inputType: OpParam, connectionExecutor: OpConnectionType): Result<OpResultType | NewOpResultType, NewOpResultErrorType> ->
                    operationExecutor(inputType, connectionExecutor) match {
                        case is ResultSuccess<OpResultType> -> $
                        case result is OpResultErrorType ->
                            executor(result, connectionExecutor)
                   }
        }

/**
* Decorates a given operation by running an additional pair of executors, one
* for successful executions and another for failed ones. These executors have
* access to the original connection and return a `Result<T, E>` value that will
* become the result of the decorated operation.
*
* This combinator is useful in the same circumstances as `onSuccessRunExecutor`
* and `onFailureRunExecutor`. This combinator exists because it can provide a
* tighter type for the resulting operation.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to decorate
* | `executors` | { onSuccessExecutor: &#40;OpResultType, OpConnectionType&#41; &#45;&#62; Result<NewOpResultType, NewOpResultErrorType&#62;, onFailureExecutor: &#40;OpResultErrorType, OpConnectionType&#41; &#45;&#62; Result<NewOpResultType, NewOpResultErrorType&#62; } | The executors to run after completed invocations of the decorated operation
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.8
*
* type AddParam = {
*   left: Number,
*   right: Number
* }
* type AddFailure = {
*   arg: Number,
*   minValid: Number,
*   maxValid: Number
* }
*
* var add: Operation<{ left: Number, right: Number }, Number, ResultFailure<AddFailure, Error>, Null> = {
*   name: "Add",
*   displayName: "Adds two numbers (result = left + right)",
*   executor: (parameter, requestBuilder) -> do {
*     var minValid = 0
*     var maxValid = 100
*     ---
*     if (maxValid < parameter.left or parameter.left < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`left` was outside the expected range")
*     else if (maxValid < parameter.right or parameter.right < minValid)
*       failure({
*         arg: parameter.left,
*         minValid: minValid,
*         maxValid: maxValid
*       }, "`right` was outside the expected range")
*     else
*       success(parameter.left + parameter.right)
*   }
* }
*
* var failsOnBigResultsAndSucceedsWithErrorCode = add
*   onCompletionRun {
*     onSuccessExecutor: (value, connection) -> do {
*         var maxResult = 100
*         ---
*         if (maxResult < value)
*           failure({
*             result: value,
*             maxResult: maxResult
*           }, "Result was too big!")
*         else
*           success(value)
*       },
*     onFailureExecutor: (value, connection) ->
*       if (value.error.value.arg < value.error.value.minValid)
*         success(-1) // Argument was too small
*       else if (value.error.value.maxValid < value.error.value.arg)
*         success(-2) // Argument was too big
*       else
*         value
*   }
* ----
*
*/
fun onCompletionRun<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpResultType, NewOpResultErrorType <: ResultFailure>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    executors: {
        onSuccessExecutor: (OpResultType, OpConnectionType) -> Result<NewOpResultType, NewOpResultErrorType>,
        onFailureExecutor: (OpResultErrorType, OpConnectionType) -> Result<NewOpResultType, NewOpResultErrorType>
    }
): Operation<OpParam, NewOpResultType, NewOpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update {
            case operationExecutor at .executor ->
                (inputType: OpParam, connectionExecutor: OpConnectionType): Result<NewOpResultType, NewOpResultErrorType> ->
                    operationExecutor(inputType, connectionExecutor) match {
                        case result is ResultSuccess<OpResultType> ->
                            executors.onSuccessExecutor(result.value, connectionExecutor)
                        case result is OpResultErrorType ->
                            executors.onFailureExecutor(result, connectionExecutor)
                   }
        }

/**
* Transform an operation into a paginated one with the aid of a pagination strategy. The next page operation will be the same one.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to paginate
* | `paginationStrategy` | &#40;OpParam, OpResultType&#41; &#45;&#62; Page<ItemType, OpParam&#62; | A callback that receives both the original parameterization and the result of a successful invocation
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* var numbers: Operation<NumbersParam, Array<Number>, Nothing, {}> = {
*   name: "numbers",
*   displayName: "Gets you a bunch of numbers",
*   executor: (params, connectionInstance) ->
*     success((params.from to (params.from + params.count)) as Array<Number>)
* }
*
* var paginatedNumbers = numbers paginated (param, page) -> {
*   items: page,
*   nextPage: { args: { from: param.from + param.count + 1, count: 10 } }
* }
* ----
*
*/
fun paginated<OpParam, OpResultType, OpResultErrorType, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    paginationStrategy: (OpParam, OpResultType) -> Page<ItemType, OpParam>
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> = do {
        var ex = (opParam: OpParam, opConnectionType: OpConnectionType) ->
                                operation.executor(opParam, opConnectionType) match {
                                    case is ResultSuccess<OpResultType> -> $ update { case value at .value -> paginationStrategy(opParam, value) }
                                    case is OpResultErrorType -> $
                                }
        ---
        if (operation.versions is Object)
            fail("You can't compose an operation with extra versions")
        else
            operation update {
                case .executor -> ex
                case .nextPageExecutor! -> ex
            }
}

/**
* Transform an operation into a paginated one with the aid of a pagination strategy and specifying the next page operation.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | The operation to paginate
* | `nextPageOperation` | Operation<OpParamNextPage, Page<ItemType, OpParamNextPage>, OpResultErrorType, OpConnectionType> | The next page operation
* | `paginationStrategy` | &#40;OpParam, OpResultType&#41; &#45;&#62; Page<ItemType, OpParam&#62; | A callback that receives both the original parameterization and the result of a successful invocation
* |===
*
*/
fun paginatedWithNextPageOp<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, ItemType, OpParamNextPage>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    nextPageOperation: Operation<OpParamNextPage, Page<ItemType, OpParamNextPage>, OpResultErrorType, OpConnectionType>,
    paginationStrategy: (OpParam, OpResultType) -> Page<ItemType, OpParamNextPage>
): PaginatedOperation<OpParam, OpParamNextPage, Page<ItemType, OpParamNextPage>, OpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update {
            case executor at .executor -> (opParam: OpParam, opConnectionType: OpConnectionType) ->
                executor(opParam, opConnectionType) match {
                    case is ResultSuccess<OpResultType> -> $ update { case value at .value -> paginationStrategy(opParam, value) }
                    case is OpResultErrorType -> $
                }
            case .nextPageExecutor! -> nextPageOperation.executor
        }

/**
* Takes an operation and maps its input and output based on the provided mappers.
* It helps to transform a raw operation with transport-specific parameters
* into a curated operation with new parameters that are faced to the platform and end user.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType> | The operation to be mapped
* | `inputMapper` | (NewOpParam) -> OpParam | Function that maps the new input parameter to the underlying input parameter
* | `outputSuccessMapper` | (OpResultType) -> NewOpResultType | A function that receives the successful result and improves it somehow
* | `outputFailureMapper` |(OpResultErrorType) -> NewOpResultErrorType | A function that receives the error and transforms it to a _better_ error
* |===
*
*/
fun mapInputAndOutputOperation<OpParam, OpResultType, OpResultErrorType <: ResultFailure, OpConnectionType, NewOpParam, NewOpResultType, NewOpResultErrorType <: ResultFailure>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    inputMapper: (NewOpParam) -> OpParam,
    outputSuccessMapper: (OpResultType) -> NewOpResultType,
    outputFailureMapper: (OpResultErrorType) -> NewOpResultErrorType
    ): Operation<NewOpParam, NewOpResultType, NewOpResultErrorType, OpConnectionType> =
    if (operation.versions is Object)
        fail("You can't compose an operation with extra versions")
    else
        operation update { case executor at .executor -> mapResult(mapInput(operation.executor, inputMapper), outputSuccessMapper, outputFailureMapper)}
