%dw 2.7
import * from com::mulesoft::connectivity::Model

/**
* Transforms an executor by applying a function that adapts its input type
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `executor` | Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType&#62; | The executor to transform
* | `mapper` | &#40;NewOpParam&#41; &#45;&#62; InputType | A function that adapts its input to a more useful one
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* var numbers: Executor<Array<Number>, Array<Number>, Nothing, {}> =
* (params, connectionInstance) -> success((params[0] to (params[0] + params[1])) as Array<Number>)
*
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* var betterNumbers = mapInput(params: NumbersParam) -> [params.from, params.count]
* ----
*/
fun mapInput<InputType, ResultType, FailureResultType <: ResultFailure, ConnectionExecutorType, NewOpParam>(
    executor: Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType>,
    mapper: (NewOpParam) -> InputType) =
    (newOpParam: NewOpParam, connectionExecutorType: ConnectionExecutorType) -> executor(mapper(newOpParam), connectionExecutorType)

/**
* Transforms an executor by applying a function on its successful results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `executor` | Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType&#62; | The executor to transform
* | `mapper` | &#40;ResultType&#41; &#45;&#62; NewOpResultType | A function that receives the successful result and improves it somehow
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* type NumbersResult = {
*   numbers: Array<Number>
* }
*
* var numbers: Executor<NumbersParam, NumbersResult, Nothing, {}> =
* (params, connectionInstance) ->
*     success({
*       numbers: (params.from to (params.from + params.count)) as Array<Number>
*     })
*
* var simplerNumbers = numbers mapSuccessResult (data) -> data.numbers
* ----
*/
fun mapSuccessResult<InputType, ResultType, FailureResultType <: ResultFailure, ConnectionExecutorType, NewOpResultType>(
    executor: Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType>,
    mapper: (ResultType) -> NewOpResultType):
    Executor<InputType, NewOpResultType, FailureResultType, ConnectionExecutorType> =
    (inputType: InputType, connectionExecutorType: ConnectionExecutorType) ->
                                                       executor(inputType, connectionExecutorType) match {
                                                           case is ResultSuccess<ResultType> -> $ update { case value at .value -> mapper(value) }
                                                           case is FailureResultType -> $
                                                       }

/**
* Transforms an executor by applying a function on its failed results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `executor` | Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType&#62; | The executor to transform
* | `mapper` | &#40;FailureResultType&#41; &#45;&#62; NewOpResultErrorType | A function that receives the error and transforms it to a _better_ error
* |===
*
* === Example
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* type NumbersParam = {
*   from: Number,
*   count: Number
* }
*
* var numbers: Executor<NumbersParam, Array<Number>, Number, {}> = (params, connectionInstance) ->
*     if (params.from > 100)
*       failure(params.from, "Number too big")
*     else
*       success((params.from to (params.from + params.count)) as Array<Number>)
*
* var betterNumbers = numbers
*   mapFailureResult (error) -> { message: "Failed because the number was too big", failedFrom: error }
* ----
*/
fun mapFailureResult<InputType, ResultType, FailureResultType <: ResultFailure, ConnectionExecutorType, NewOpResultErrorType>(
    executor: Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType>,
    mapper: (FailureResultType) -> NewOpResultErrorType)
    : Executor<InputType, ResultType, NewOpResultErrorType, ConnectionExecutorType> =
    (inputType: InputType, connectionExecutorType: ConnectionExecutorType) ->
            executor(inputType, connectionExecutorType) match {
                case is ResultSuccess<ResultType> -> $
                case is FailureResultType -> mapper($)
}

/**
* Transforms an executor by applying a function on its successful and failed results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `executor` | Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType&#62; | The executor to transform
* | `successMapper` | &#40;ResultType&#41; &#45;&#62; NewOpResultType | A function that receives the successful result and improves it somehow
* | `failureMapper` | &#40;FailureResultType&#41; &#45;&#62; NewFailureResultType | A function that receives the error and transforms it to a _better_ error
* |===
*/
fun mapResult<InputType, ResultType, FailureResultType <: ResultFailure, ConnectionExecutorType, NewOpResultType, NewFailureResultType>
  (executor: Executor<InputType, ResultType, FailureResultType, ConnectionExecutorType>,
  successMapper: (ResultType) -> NewOpResultType,
  failureMapper: (FailureResultType) -> NewFailureResultType)
    : Executor<InputType, NewOpResultType, NewFailureResultType, ConnectionExecutorType> =
  (inputType: InputType, connectionExecutorType: ConnectionExecutorType) ->
                                                     executor(inputType, connectionExecutorType) match {
                                                         case is ResultSuccess<ResultType> -> $ update { case value at .value -> successMapper(value) }
                                                         case is FailureResultType -> failureMapper($)
                                                     }
