%dw 2.7
/**
* This module provides utilities to aid LinkWeave integrations by platform adapters
*/
import fromBase64 from dw::core::Binaries
import toBase64 from dw::core::Binaries
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Trigger

/******************************************************************************************************************
 * Operation Execution
 *******************************************************************************************************************/
/**
* Executes an operation.
*
* The result of successful executions can potentially be used to retrieve further pages. See `com::mulesoft::connectivity::Model::Page`
* for more info.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<InputType, ResultType, ResultErrorType, ConnectionType&#62; | The operation to execute
* | `parameters` | InputType | The parameters for the invocation
* | `connection` | ConnectionType | The connection used for the invocation
* |===
*
*/
fun executeOperation<InputType <: Object, ResultType, ResultErrorType <: ResultFailure, ConnectionType>(
    operation: Operation<InputType, ResultType, ResultErrorType, ConnectionType>, parameters: InputType, connection: ConnectionType
): Result<ResultType, ResultErrorType> =
   operation.executor(parameters, connection)

fun executePaginatedOperationNextPage<InputType <: Object, NextPageInputType <: Object, ItemType, ResultErrorType <: ResultFailure, ConnectionType>(
    operation: PaginatedOperation<InputType, NextPageInputType, Page<ItemType, NextPageInputType>, ResultErrorType, ConnectionType>,
    parameters: NextPageInputType,
    connection: ConnectionType
): Result<Page<ItemType, NextPageInputType>, ResultErrorType> =
    operation.nextPageExecutor(parameters, connection)


/******************************************************************************************************************
 * Value Provider Execution
 *******************************************************************************************************************/
 fun executeValueProvider<InputType <: Object, ErrorType, ProvidedValueType, DisplayPropertiesType <: Object, ConnectionType>(
     valueProvider: ValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType, ConnectionType>, parameters: InputType, connection: ConnectionType)
     : Result = valueProvider(parameters, connection)

fun executePaginatedValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType <: Object, ConnectionType>(
    valueProvider: PaginatedValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType, ConnectionType>,
    parameters: InputType,
    connection: ConnectionType
): Result<Page<ProvidedValue<ProvidedValueType, DisplayPropertiesType>, InputType>, ResultFailure<ErrorType, Error>> = valueProvider.executor(parameters, connection)

fun executePaginatedValueProviderNextPage<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType <: Object, ConnectionType>(
    valueProvider: PaginatedValueProvider<InputType, ErrorType, ProvidedValueType, DisplayPropertiesType, ConnectionType>,
    parameters: InputType,
    connection: ConnectionType
): Result<Page<ProvidedValue<ProvidedValueType, Object>, InputType>, ResultFailure<ErrorType, Error>> = valueProvider.nextPageExecutor(parameters, connection)

/******************************************************************************************************************
 * Metadata Provider Execution
 *******************************************************************************************************************/

fun executeMetadataProvider<InputType <: Object, ErrorType, ConnectionType>(
    metadataProvider: MetadataProviderExecutor<InputType, ErrorType, ConnectionType>,
    parameters: InputType,
    connection: ConnectionType) : Result =
    metadataProvider(parameters, connection)

/******************************************************************************************************************
 * Trigger Execution
 *******************************************************************************************************************/
/**
* Executes a trigger
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> | The trigger to execute the poll
* | `latestWatermark` | Watermark | Null | Latest watermark processed
* | `params` | TriggerInputType | OperationInputType | The parameters for the invocation
* | `connection` | ConnectionType | The connection used for the invocation
* |===
*
*/
fun executeTrigger<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType <: ResultFailure, ConnectionType, TriggerResultItemType,
 OperationResultItemType, OperationNextPageInputType, Watermark>(
    trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
    latestWatermark: Watermark,
    params: TriggerInputType,
    connection: ConnectionType
): Result<TriggerPage<OperationNextPageInputType, TriggerResultItemType,Watermark> |
TriggerPage<OperationInputType, TriggerResultItemType,Watermark>, ResultErrorType> =
    executePolling<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType, TriggerResultItemType ,
        OperationResultItemType, OperationNextPageInputType, Watermark, ConnectionType>
        (trigger, params, latestWatermark, connection)



/**
* Executes a trigger next page, same polling context
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<Any, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> | The trigger to execute the poll
* | `latestWatermark` | Watermark | Null | Latest watermark processed
* | `partialGreatestWatermark` | Watermark | Null | The greatest watermark gotten from the page or previous partialGreatestWatermark from the input
* | `params` | TriggerInput<OperationInputType, Watermark> | The parameters for the invocation
* | `connection` | ConnectionType | The connection used for the invocation
* |===
*
*/
fun executeTriggerNextPage<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType <: ResultFailure, ConnectionType, TriggerResultItemType,
 OperationResultItemType, OperationNextPageInputType, Watermark>(
    trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
    latestWatermark: Watermark,
    partialGreatestWatermark: Watermark,
    params: OperationNextPageInputType,
    connection: ConnectionType
): Result<TriggerPage<OperationNextPageInputType, TriggerResultItemType,Watermark> |
TriggerPage<OperationInputType, TriggerResultItemType,Watermark>, ResultErrorType> =
        executePollingNextPage<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType, TriggerResultItemType ,OperationResultItemType,Watermark, OperationNextPageInputType, ConnectionType>
        (trigger, params, {latestWatermark:latestWatermark, greatestWatermark: partialGreatestWatermark}, connection)

/**
* Compare trigger item's watermark
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<Any, OperationInputType, OperationOutputType,OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> |
*               The trigger which has the watermark compareTo method
* | `watermarkA` | Watermark | Previous watermark to be compared
* | `watermarkB` | Watermark | New watermark to be compared
* |===
*
*/
fun executeTriggerCompareWatermark<OperationInputType, OperationOutputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, OperationNextPageInputType, Watermark>(
    trigger: Trigger<Any, OperationInputType, OperationOutputType,OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
    watermarkA: Watermark,
    watermarkB: Watermark
): Number =
   trigger.strategy.watermarkCompareTo(watermarkA,watermarkB)

/**
* Get trigger's initial watermark
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<Any, OperationInputType, OperationOutputType,OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> |
*               The trigger which has the initialWatermark get method
* |===
*
*/
fun getInitialWatermark<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, OperationNextPageInputType, Watermark>(
    trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType,OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
    triggerInput: TriggerInputType,
    connection: ConnectionType
): Watermark =
        trigger.initialWatermark(triggerInput, connection)

/**
* Deserialize from base 64 object string
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `object` | Any | The object string to deserialize. it should be String | Null
* |===
*
*/
fun deserializeObject(object) =
    if(object != null) read(fromBase64(object) as Binary {encoding: "UTF-8"},'application/dw', { onlyData: true }) else null

/**
* Serialize to base 64 object
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `object` | Any | The object to serialize. it should be Object | Null
* |===
*
*/
fun serializeObject(object)=
    if(object != null) toBase64(write(object,'application/dw',{ignoreSchema: true}) as Binary {encoding: "UTF-8"}) else null

/**
* Serialize to base 64 success trigger result
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `result` | Result<TriggerPage,ResultErrorType> | The Result object to serialize.
* |===
*
*/
fun serializeTriggerResponseIfSuccess<ResultErrorType>(result: Result<TriggerPage,ResultErrorType>) =
    if(result.success)
        result update {
            case .value ->  $ update {
                case .nextPoll.latestWatermark -> serializeObject($)
                case .nextPoll.greatestWatermark -> serializeObject($)
                case .nextPage -> serializeObject($)
            }
        }
    else
        result
