/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.util;

import static java.lang.String.format;
import static java.nio.file.Files.newInputStream;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.apache.commons.io.IOUtils.toByteArray;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class FileUtils {

  private static final String META_INF = "META-INF";
  private static final String MAVEN_PATH = META_INF + "/maven";
  private static final String MULE_PLUGIN_POM = "pom.xml";

  /**
   * Finds the URL of the pom file within the artifact file.
   *
   * @param artifactFile the artifact file to search for the pom file.
   * @return the URL to the pom file.
   */
  public static URL getPomUrlFromJar(Path artifactFile) {
    try {
      List<URL> jarMavenUrls = getUrlsWithinJar(artifactFile, MAVEN_PATH);
      if (jarMavenUrls.isEmpty()) {
        throw new RuntimeException(format("No entries found in %s for artifact %s",
                                          MAVEN_PATH,
                                          artifactFile.getFileName().toString()));
      }
      Optional<URL> pomUrl = jarMavenUrls.stream().filter(url -> url.toString().endsWith(MULE_PLUGIN_POM)).findAny();
      if (!pomUrl.isPresent()) {
        String expectedPath = MAVEN_PATH + "/${groupId}/${artifactId}/" + MULE_PLUGIN_POM;
        throw new RuntimeException(format("The file '%s' was missing while looking within the artifact %s (it should be placed under [%s]).%s"
            + "Found URLs in Maven folder: %s",
                                          MULE_PLUGIN_POM,
                                          artifactFile.getFileName().toString(),
                                          expectedPath,
                                          jarMavenUrls));
      }
      return pomUrl.get();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Loads the content of a file within a jar into a byte array.
   *
   * @param jarFile the jar file
   * @return the content of the file as byte array or empty if the file does not exists within the jar file.
   * @throws IOException if there was a problem reading from the jar file.
   */
  public static Optional<byte[]> loadFileContentFrom(URL jarFile) throws IOException {
    JarURLConnection jarConnection = (JarURLConnection) jarFile.openConnection();
    jarConnection.setUseCaches(false);

    try (InputStream stream = jarConnection.getInputStream()) {
      return of(toByteArray(stream));
    } catch (FileNotFoundException e) {
      return empty();
    }
  }

  /**
   * Creates an URL to a path within a jar file.
   *
   * @param jarFile  the jar file
   * @param filePath the path within the jar file
   * @return an URL to the {@code filePath} within the {@code jarFile}
   * @throws MalformedURLException if the provided {@code filePath} is malformed
   */
  private static URL getUrlWithinJar(Path jarFile, String filePath) throws MalformedURLException {
    return new URL("jar:" + jarFile.toUri().toString() + "!/" + filePath);
  }

  /**
   * Gets all the URL of files within a directory
   *
   * @param file      the jar file
   * @param directory the directory within the jar file (using "/" as path separator).
   * @return a collection of URLs to files within the directory {@code directory}. Empty collection if the directory does not
   *         exists or is empty.
   * @throws IOException
   */
  private static List<URL> getUrlsWithinJar(Path file, String directory) throws IOException {
    List<URL> urls = new ArrayList<>();

    try (ZipInputStream zis = new ZipInputStream(newInputStream(file))) {
      ZipEntry jarEntry = zis.getNextEntry();
      while (jarEntry != null) {
        String entryName = jarEntry.getName();

        // Validate entry path to prevent Zip Slip vulnerability
        if (!isValidJarEntryPath(entryName)) {
          throw new IOException("Invalid jar entry path detected: " + entryName);
        }

        if (!jarEntry.isDirectory() && entryName.startsWith(directory + "/")) {
          urls.add(getUrlWithinJar(file, entryName));
        }

        jarEntry = zis.getNextEntry();
      }
    }

    return urls;
  }

  /**
   * Validates a jar entry path to prevent path traversal attacks (Zip Slip).
   *
   * @param entryPath the jar entry path to validate
   * @return true if the path is valid, false otherwise
   */
  private static boolean isValidJarEntryPath(String entryPath) {
    if (entryPath == null || entryPath.isEmpty()) {
      return false;
    }

    // Check for path traversal sequences
    if (entryPath.contains("..")) {
      return false;
    }

    // Check for absolute paths (should be relative within jar)
    if (entryPath.startsWith("/") || entryPath.startsWith("\\")) {
      return false;
    }

    // Normalize and check that the path doesn't escape
    String normalizedPath = new File(entryPath).toPath().normalize().toString();
    return !normalizedPath.startsWith("..");
  }

  /**
   * Workaround for JDK bug <a href="http://bugs.sun.com/bugdatabase/view_bug.do;:YfiG?bug_id=4117557"> 4117557</a>. More
   * in-context information at <a href="http://mule.mulesoft.org/jira/browse/MULE-1112">MULE-1112</a>
   * <p/>
   * Factory methods correspond to constructors of the <code>java.io.File class</code>. No physical file created in this method.
   *
   * @see File
   */
  public static File newFile(String pathName) {
    try {
      return new File(pathName).getCanonicalFile();
    } catch (IOException e) {
      throw new RuntimeException("Unable to create a canonical file for " + pathName, e);
    }
  }

}
