/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal;

import static org.mule.maven.pom.parser.api.model.BundleScope.valueOf;
import static org.mule.maven.pom.parser.internal.util.MavenUtils.getPomModel;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.util.Collections.emptyList;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.regex.Pattern.compile;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Stream.concat;

import static org.codehaus.plexus.util.StringUtils.isEmpty;
import static org.codehaus.plexus.util.xml.Xpp3DomUtils.mergeXpp3Dom;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.SharedLibrary;
import org.mule.maven.pom.parser.internal.model.MavenPomModelWrapper;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import org.apache.maven.model.Build;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.FileSet;
import org.apache.maven.model.Model;
import org.apache.maven.model.Parent;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginManagement;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import org.slf4j.Logger;

public class MavenPomParserImpl implements MavenPomParser {

  private static final Logger LOGGER = getLogger(MavenPomParserImpl.class);

  private final Pattern MAVEN_PROPERTY_PATTERN = compile("(\\$\\{[^}]+})");

  private static final String MULE_MAVEN_PLUGIN_GROUP_ID = "org.mule.tools.maven";
  private static final String MULE_MAVEN_PLUGIN_ARTIFACT_ID = "mule-maven-plugin";
  private static final String MULE_EXTENSIONS_PLUGIN_GROUP_ID = "org.mule.runtime.plugins";
  private static final String MULE_EXTENSIONS_PLUGIN_ARTIFACT_ID = "mule-extensions-maven-plugin";
  private static final String GROUP_ID = "groupId";
  private static final String ARTIFACT_ID = "artifactId";
  public static final String SHARED_LIBRARIES_FIELD = "sharedLibraries";
  public static final String SHARED_LIBRARY_FIELD = "sharedLibrary";
  public static final String DEFAULT_SOURCES_DIRECTORY = "src/main";
  private static final String VERSION = "version";
  private static final String MAVEN_SHADE_PLUGIN_ARTIFACT_ID = "maven-shade-plugin";
  private static final String ORG_APACHE_MAVEN_PLUGINS_GROUP_ID = "org.apache.maven.plugins";
  private static final String ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD = "additionalPluginDependencies";
  private static final String PLUGIN_FIELD = "plugin";
  private static final String PLUGIN_DEPENDENCY_FIELD = "dependency";
  private static final String PLUGIN_DEPENDENCIES_FIELD = "additionalDependencies";
  private static final String DEFAULT_ARTIFACT_TYPE = "jar";
  private final Model pomModel;
  private final List<String> activeProfiles;
  private final Build build;

  public MavenPomParserImpl(Path path) {
    this(path, emptyList());
  }

  public MavenPomParserImpl(Path path, List<String> activeProfiles) {
    requireNonNull(path, "path cannot be null");
    this.pomModel = getPomModel(path.toFile());
    this.activeProfiles = activeProfiles;
    this.build = pomModel.getBuild();
  }

  @Override
  public String getSourceDirectory() {
    return build != null && build.getSourceDirectory() != null ? build.getSourceDirectory() : DEFAULT_SOURCES_DIRECTORY;
  }

  @Override
  public List<String> getResourceDirectories() {
    return build != null ? build.getResources().stream().map(FileSet::getDirectory).collect(toList()) : emptyList();
  }

  @Override
  public MavenPomModel getModel() {
    return new MavenPomModelWrapper(pomModel);
  }

  @Override
  public List<SharedLibrary> getSharedLibraries() {
    return findArtifactPackagerPlugin().stream()
        .flatMap(this::getSharedLibrariesDom)
        .findFirst()
        .map(Stream::of)
        .orElseGet(Stream::empty)
        .map(sharedLibraryDom -> new SharedLibrary(
                                                   getAttribute(sharedLibraryDom, GROUP_ID),
                                                   getAttribute(sharedLibraryDom, ARTIFACT_ID)))
        .collect(collectingAndThen(toList(), Collections::unmodifiableList));
  }

  private Stream<Xpp3Dom[]> getSharedLibrariesDom(Plugin plugin) {
    Object configuration = plugin.getConfiguration();
    if (configuration != null) {
      Xpp3Dom sharedLibrariesDom = ((Xpp3Dom) configuration).getChild(SHARED_LIBRARIES_FIELD);
      if (sharedLibrariesDom != null) {
        return Stream.<Xpp3Dom[]>of(sharedLibrariesDom.getChildren(SHARED_LIBRARY_FIELD));
      }
    }

    return Stream.empty();
  }

  @Override
  public List<BundleDependency> getDependencies() {
    List<Dependency> dependencies = pomModel.getDependencies();
    Optional<Model> parentModel = loadParentModel(pomModel);
    while (parentModel.isPresent()) {
      dependencies.addAll(parentModel.get().getDependencies().stream().filter(dependency -> dependencies.stream()
          .noneMatch(d -> d.getGroupId().equals(dependency.getGroupId()) && d.getArtifactId().equals(dependency.getArtifactId())
              && d.getClassifier().equals(dependency.getClassifier()) && d.getType().equals(dependency.getType())))
          .collect(toList()));
      parentModel = loadParentModel(parentModel.get());
    }

    return dependencies.stream().map(this::toBundleDependency).collect(toList());
  }

  @Override
  public Map<ArtifactCoordinates, AdditionalPluginDependencies> getPomAdditionalPluginDependenciesForArtifacts() {
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pluginsAdditionalLibraries = new HashMap<>();
    findArtifactPackagerPlugin().stream()
        .flatMap(this::getAdditionalPluginDependenciesPluginsDom)
        .findFirst()
        .map(Stream::of)
        .orElseGet(Stream::empty)
        .forEach(pluginDom -> {
          String pluginGroupId = getChildParameterValue(pluginDom, GROUP_ID, true);
          String pluginArtifactId = getChildParameterValue(pluginDom, ARTIFACT_ID, true);
          List<BundleDescriptor> additionalDependencyDependencies = new ArrayList<>();
          Xpp3Dom dependenciesDom = pluginDom.getChild(PLUGIN_DEPENDENCIES_FIELD);
          if (dependenciesDom != null) {
            for (Xpp3Dom dependencyDom : dependenciesDom.getChildren(PLUGIN_DEPENDENCY_FIELD)) {
              BundleDescriptor.Builder dependency = new BundleDescriptor.Builder();
              dependency.setGroupId(getChildParameterValue(dependencyDom, GROUP_ID, true));
              dependency
                  .setArtifactId(getChildParameterValue(dependencyDom, ARTIFACT_ID, true));
              dependency.setVersion(getChildParameterValue(dependencyDom, VERSION, true));
              String type = getChildParameterValue(dependencyDom, "type", false);
              dependency.setType(type == null ? DEFAULT_ARTIFACT_TYPE : type);
              dependency.setClassifier(getChildParameterValue(dependencyDom, "classifier", false));
              dependency.setSystemPath(getChildParameterValue(dependencyDom, "systemPath", false));

              additionalDependencyDependencies.add(dependency.build());
            }
          }
          AdditionalPluginDependencies additionalPluginDependencies =
              new AdditionalPluginDependencies(pluginGroupId, pluginArtifactId, additionalDependencyDependencies);
          pluginsAdditionalLibraries.put(new ArtifactCoordinates(pluginGroupId, pluginArtifactId),
                                         additionalPluginDependencies);
        });

    return pluginsAdditionalLibraries;
  }

  private Stream<Xpp3Dom[]> getAdditionalPluginDependenciesPluginsDom(Plugin plugin) {
    Object configuration = plugin.getConfiguration();
    if (configuration != null) {
      Xpp3Dom additionalPluginDependenciesDom = ((Xpp3Dom) configuration).getChild(ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD);
      if (additionalPluginDependenciesDom != null) {
        return Stream.<Xpp3Dom[]>of(additionalPluginDependenciesDom.getChildren(PLUGIN_FIELD));
      }
    }

    return Stream.empty();
  }

  @Override
  public Properties getProperties() {
    Properties properties = pomModel.getProperties();
    Optional<Model> parentModel = loadParentModel(pomModel);
    while (parentModel.isPresent()) {
      properties.putAll(parentModel.get().getProperties().entrySet().stream().filter(property -> properties.keySet().stream()
          .noneMatch(key -> key.equals(property.getKey())))
          .collect(toMap(Entry::getKey, Entry::getValue)));
      parentModel = loadParentModel(parentModel.get());
    }

    return properties;
  }

  @Override
  public boolean isMavenShadePluginConfigured() {
    if (pomModel.getBuild() != null) {
      for (Plugin plugin : pomModel.getBuild().getPlugins()) {
        if (plugin.getGroupId().equals(ORG_APACHE_MAVEN_PLUGINS_GROUP_ID)
            && plugin.getArtifactId().equals(MAVEN_SHADE_PLUGIN_ARTIFACT_ID)) {
          return true;
        }
      }
    }

    return false;
  }

  private List<Plugin> findArtifactPackagerPlugin() {
    Stream<Plugin> basePlugin = Stream.empty();
    Optional<Plugin> pluginManagementPlugin = empty();
    Build build = pomModel.getBuild();
    if (build != null) {
      basePlugin = findArtifactPackagerPlugin(build.getPlugins()).map(Stream::of).orElse(Stream.empty());
      pluginManagementPlugin = findArtifactPackagerPlugin(build.getPluginManagement());
    }

    // Sort them so the processing is consistent with how Maven calculates the plugin configuration for the effective pom.
    final List<String> sortedActiveProfiles = activeProfiles
        .stream()
        .sorted(String::compareTo)
        .collect(toList());

    final Stream<Plugin> packagerConfigsForActivePluginsStream = pomModel.getProfiles().stream()
        .filter(profile -> sortedActiveProfiles.contains(profile.getId()))
        .map(profile -> findArtifactPackagerPlugin(profile.getBuild() != null ? profile.getBuild().getPlugins() : null))
        .filter(plugin -> !plugin.equals(empty()))
        .map(Optional::get);

    Optional<Plugin> plugin = concat(basePlugin, packagerConfigsForActivePluginsStream)
        .reduce((p1, p2) -> {
          p1.setConfiguration(mergeXpp3Dom((Xpp3Dom) p2.getConfiguration(), (Xpp3Dom) p1.getConfiguration()));
          p1.getDependencies().addAll(p2.getDependencies());

          return p1;
        });

    List<Plugin> plugins = new ArrayList<>();
    plugin.ifPresent(plugins::add);
    pluginManagementPlugin.ifPresent(plugins::add);

    // Check parent POMs for the packager plugin - collect all plugins from parent hierarchy
    loadParentModel(pomModel).ifPresent(pm -> plugins.addAll(findArtifactPackagerPluginInParents(pm)));

    return plugins;
  }

  /**
   * Recursively searches for the artifact packager plugin in parent POMs. Collects all matching plugins from the entire parent
   * hierarchy.
   *
   * @param parentModel the parent model to search
   * @return List of all matching plugins found in the parent hierarchy
   */
  private List<Plugin> findArtifactPackagerPluginInParents(Model parentModel) {
    List<Plugin> plugins = new ArrayList<>();

    // Check current parent model for the plugin
    if (parentModel.getBuild() != null) {
      findArtifactPackagerPlugin(parentModel.getBuild().getPlugins()).ifPresent(plugins::add);
      findArtifactPackagerPlugin(parentModel.getBuild().getPluginManagement()).ifPresent(plugins::add);
    }

    // Continue searching in the parent's parent
    loadParentModel(parentModel).ifPresent(pm -> plugins.addAll(findArtifactPackagerPluginInParents(pm)));

    return plugins;
  }

  private Optional<Plugin> findArtifactPackagerPlugin(PluginManagement pluginManagement) {
    if (pluginManagement != null && pluginManagement.getPlugins() != null) {
      return findArtifactPackagerPlugin(pluginManagement.getPlugins());
    }

    return empty();
  }

  private Optional<Plugin> findArtifactPackagerPlugin(List<Plugin> plugins) {
    if (plugins == null) {
      return empty();
    }

    return plugins.stream().filter(plugin -> (plugin.getArtifactId().equals(MULE_MAVEN_PLUGIN_ARTIFACT_ID)
        && plugin.getGroupId().equals(MULE_MAVEN_PLUGIN_GROUP_ID)) ||
        (plugin.getArtifactId().equals(MULE_EXTENSIONS_PLUGIN_ARTIFACT_ID) &&
            plugin.getGroupId().equals(MULE_EXTENSIONS_PLUGIN_GROUP_ID)))
        .findFirst();
  }

  private String getAttribute(Xpp3Dom tag, String attributeName) {
    Xpp3Dom attributeDom = tag.getChild(attributeName);
    checkState(attributeDom != null, format("'%s' element not declared at '%s' in the pom file",
                                            attributeName, tag));
    String attributeValue = attributeDom.getValue().trim();
    checkState(!isEmpty(attributeValue),
               format("'%s' was defined but has an empty value at '%s' declared in the pom file",
                      attributeName, tag));
    return attributeValue;
  }

  private String getChildParameterValue(Xpp3Dom element, String childName, boolean validate) {
    Xpp3Dom child = element.getChild(childName);
    String childValue = child != null ? child.getValue() : null;
    if (isEmpty(childValue) && validate) {
      throw new IllegalArgumentException("Expecting child element with not null value " + childName);
    }

    if (isEmpty(childValue)) {
      return childValue;
    }

    int lastIndex = 0;
    StringBuilder output = new StringBuilder();
    final Matcher matcher = MAVEN_PROPERTY_PATTERN
        .matcher(childValue);
    while (matcher.find()) {
      final String propertyValue = resolvePropertyValue(matcher.group(1).substring(2, matcher.group(1).length() - 1));
      if (propertyValue != null) {
        output.append(childValue, lastIndex, matcher.start())
            .append(propertyValue);
      } else {
        output.append(childValue, lastIndex, matcher.start())
            .append(matcher.group(1));
      }

      lastIndex = matcher.end();
    }
    if (lastIndex < childValue.length()) {
      output.append(childValue, lastIndex, childValue.length());
    }

    return output.toString();
  }

  private String resolvePropertyValue(final String propertyName) {
    String propertyValue = getProperty(propertyName);
    if (propertyValue == null) {
      propertyValue = getProperties().getProperty(propertyName);
    }
    return propertyValue;
  }

  private BundleDependency toBundleDependency(Dependency dependency) {
    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(dependency.getGroupId())
        .setArtifactId(dependency.getArtifactId())
        .setVersion(dependency.getVersion())
        .setType(dependency.getType())
        .setExclusions(dependency.getExclusions().stream().map(exclusion -> new ArtifactCoordinates(exclusion.getGroupId(),
                                                                                                    exclusion.getArtifactId()))
            .collect(toList()))
        .setClassifier(dependency.getClassifier())
        .setOptional(dependency.getOptional())
        .setSystemPath(dependency.getSystemPath())
        .build();
    BundleDependency.Builder builder = new BundleDependency.Builder().setDescriptor(descriptor);
    if (dependency.getScope() != null && !dependency.getScope().isEmpty()) {
      builder.setScope(valueOf(dependency.getScope().toUpperCase()));
    }
    return builder.build();
  }

  /**
   * Loads the parent POM model if it exists.
   *
   * @param model the current model
   * @return Optional containing the parent Model if it exists and can be loaded
   */
  // TODO W-20138577 - Resolve parent POM when not available in project
  private Optional<Model> loadParentModel(Model model) {
    Parent parent = model.getParent();
    if (parent == null) {
      return empty();
    }

    // Try to find the parent POM file
    File currentPomFile = model.getPomFile();
    if (currentPomFile == null) {
      return empty();
    }

    // Look for parent pom in the relative path specified
    String relativePath = parent.getRelativePath();
    if (relativePath == null || relativePath.isEmpty()) {
      relativePath = "../pom.xml";
    }

    File parentPomFile = new File(currentPomFile.getParentFile(), relativePath);
    if (!parentPomFile.exists()) {
      return empty();
    }

    // If the relative path points to a directory, look for pom.xml inside it
    if (parentPomFile.isDirectory()) {
      parentPomFile = new File(parentPomFile, "pom.xml");
    }

    if (!parentPomFile.exists()) {
      return empty();
    }

    try {
      return of(getPomModel(parentPomFile));
    } catch (Exception e) {
      LOGGER.debug("Couldn't load parent model from '{}' for '{}'", parentPomFile, model);
      return empty();
    }
  }

  private void checkState(boolean condition, String message) {
    if (!condition) {
      throw new IllegalStateException(message);
    }
  }
}
