/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.model;

import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.MAVEN_PARSER_CLIENT;
import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.PomBuildingStory.MAVEN_POM_BUILDING_STORY;
import static org.mule.maven.pom.parser.internal.model.MavenModelBuilderImpl.MULE_POM_PROPERTIES;
import static org.mule.maven.pom.parser.internal.util.TestUtils.getPath;

import static java.nio.file.Paths.get;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.io.FileMatchers.anExistingFile;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenModelBuilder;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.SharedLibrary;
import org.mule.maven.pom.parser.internal.MavenPomParserImpl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

@Feature(MAVEN_PARSER_CLIENT)
@Story(MAVEN_POM_BUILDING_STORY)
public class MavenModelBuilderTestCase {

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private final static String GROUP_ID = "groupId";
  private final static String ARTIFACT_ID = "artifactId";
  private final static String VERSION = "1.0.0-SNAPSHOT";
  private final static String PACKAGING = "mule-application";
  private final static String BASIC_POM = "basic/pom.xml";
  private final static String SHARED_LIB_POM = "shared-lib/pom.xml";
  private final static String ADDITIONAL_PLUGIN_DEPENDENCIES_POM = "additional-plugin-dependency/pom.xml";
  private final static String DERBY_GROUP_ID = "org.apache.derby";
  private final static String DERBY_ARTIFACT_ID = "derby";
  private final static String DERBY_VERSION = "10.14.2.0";
  private final static String PROPERTY_KEY = "key";
  private final static String PROPERTY_VALUE = "value";
  private static final BundleDescriptor DERBY_DESCRIPTOR = new BundleDescriptor.Builder()
      .setGroupId(DERBY_GROUP_ID).setArtifactId(DERBY_ARTIFACT_ID)
      .setVersion(DERBY_VERSION)
      .build();

  @Test
  public void createPomTestCase() {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(GROUP_ID, ARTIFACT_ID, VERSION, empty(), of(PACKAGING));
    MavenPomModel model = mavenModelBuilder.getModel();
    assertThat(model.getGroupId(), is(GROUP_ID));
    assertThat(model.getArtifactId(), is(ARTIFACT_ID));
    assertThat(model.getVersion(), is(VERSION));
    assertThat(model.getPackaging(), is(PACKAGING));
  }

  @Test
  public void createPomFromFile() {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(BASIC_POM));
    MavenPomModel model = mavenModelBuilder.getModel();
    assertThat(model.getGroupId(), is("com.test"));
    assertThat(model.getArtifactId(), is("basic"));
    assertThat(model.getVersion(), is(VERSION));
    assertThat(model.getPackaging(), is(PACKAGING));
  }

  @Test
  public void addAdditionalDependencyTestCase() {
    AdditionalPluginDependencies additionalPluginDependencies =
        new AdditionalPluginDependencies("org.mule.connectors", "mule-db-connector", singletonList(DERBY_DESCRIPTOR));
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(BASIC_POM));
    mavenModelBuilder.addAdditionalPluginDependency(additionalPluginDependencies);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pomAdditionalPluginDependenciesForArtifacts =
        parser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(pomAdditionalPluginDependenciesForArtifacts.entrySet(), hasSize(1));
    AdditionalPluginDependencies pluginDependencies =
        pomAdditionalPluginDependenciesForArtifacts.get(new ArtifactCoordinates("org.mule.connectors", "mule-db-connector"));
    List<BundleDescriptor> additionalDependencies = pluginDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(1));
    BundleDescriptor derbyDescriptor = additionalDependencies.get(0);
    assertThat(derbyDescriptor.getGroupId(), is(DERBY_GROUP_ID));
    assertThat(derbyDescriptor.getArtifactId(), is(DERBY_ARTIFACT_ID));
    assertThat(derbyDescriptor.getVersion(), is(DERBY_VERSION));
  }

  @Test
  public void addAdditionalDependencyToPomWithAdditionalDependenciesForTheSamePluginTestCase() {
    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID).setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .build();
    AdditionalPluginDependencies additionalPluginDependencies =
        new AdditionalPluginDependencies("org.mule.connectors", "mule-db-connector", singletonList(descriptor));
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(ADDITIONAL_PLUGIN_DEPENDENCIES_POM));
    mavenModelBuilder.addAdditionalPluginDependency(additionalPluginDependencies);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pomAdditionalPluginDependenciesForArtifacts =
        parser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(pomAdditionalPluginDependenciesForArtifacts.entrySet(), hasSize(1));
    AdditionalPluginDependencies pluginDependencies =
        pomAdditionalPluginDependenciesForArtifacts.get(new ArtifactCoordinates("org.mule.connectors", "mule-db-connector"));
    List<BundleDescriptor> additionalDependencies = pluginDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(2));
    assertThat(additionalDependencies, containsInAnyOrder(descriptor, DERBY_DESCRIPTOR));
  }

  @Test
  public void addAdditionalDependencyToPomWithAdditionalDependenciesForOtherPluginTestCase() {
    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID).setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .build();
    AdditionalPluginDependencies additionalPluginDependencies =
        new AdditionalPluginDependencies("org.mule.connectors", "mule-http-connector", singletonList(descriptor));
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(ADDITIONAL_PLUGIN_DEPENDENCIES_POM));
    mavenModelBuilder.addAdditionalPluginDependency(additionalPluginDependencies);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pomAdditionalPluginDependenciesForArtifacts =
        parser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(pomAdditionalPluginDependenciesForArtifacts.entrySet(), hasSize(2));
    assertThat(pomAdditionalPluginDependenciesForArtifacts,
               hasKey(new ArtifactCoordinates("org.mule.connectors", "mule-db-connector")));
    ArtifactCoordinates httpConnectorArtifact = new ArtifactCoordinates("org.mule.connectors", "mule-http-connector");
    assertThat(pomAdditionalPluginDependenciesForArtifacts, hasKey(httpConnectorArtifact));
    AdditionalPluginDependencies pluginDependencies = pomAdditionalPluginDependenciesForArtifacts.get(httpConnectorArtifact);
    List<BundleDescriptor> additionalDependencies = pluginDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(1));
    assertThat(additionalDependencies, containsInAnyOrder(descriptor));
  }

  @Test
  public void addRepeatedAdditionalDependencyWithSameVersionTestCase() {
    AdditionalPluginDependencies additionalPluginDependencies =
        new AdditionalPluginDependencies("org.mule.connectors", "mule-db-connector", singletonList(DERBY_DESCRIPTOR));
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(ADDITIONAL_PLUGIN_DEPENDENCIES_POM));
    mavenModelBuilder.addAdditionalPluginDependency(additionalPluginDependencies);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pomAdditionalPluginDependenciesForArtifacts =
        parser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(pomAdditionalPluginDependenciesForArtifacts.entrySet(), hasSize(1));
    AdditionalPluginDependencies pluginDependencies =
        pomAdditionalPluginDependenciesForArtifacts.get(new ArtifactCoordinates("org.mule.connectors", "mule-db-connector"));
    List<BundleDescriptor> additionalDependencies = pluginDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(1));
    assertThat(additionalDependencies, containsInAnyOrder(DERBY_DESCRIPTOR));
  }

  @Test(expected = RuntimeException.class)
  public void addRepeatedAdditionalDependencyWithDifferentVersionTestCase() {
    BundleDescriptor otherDerbyVersion = new BundleDescriptor.Builder()
        .setGroupId(DERBY_GROUP_ID).setArtifactId(DERBY_ARTIFACT_ID)
        .setVersion("1.0.0")
        .build();
    AdditionalPluginDependencies additionalPluginDependencies =
        new AdditionalPluginDependencies("org.mule.connectors", "mule-db-connector", singletonList(otherDerbyVersion));
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(ADDITIONAL_PLUGIN_DEPENDENCIES_POM));
    mavenModelBuilder.addAdditionalPluginDependency(additionalPluginDependencies);
  }

  @Test
  public void addShareLibDependencyTestCase() {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(BASIC_POM));
    mavenModelBuilder.addSharedLibraryDependency(DERBY_GROUP_ID, DERBY_ARTIFACT_ID);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    List<SharedLibrary> sharedLibraries = parser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is(DERBY_GROUP_ID));
    assertThat(sharedLibrary.getArtifactId(), is(DERBY_ARTIFACT_ID));
  }

  @Test
  public void addAlreadyAddedShareLibDependencyTestCase() {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(SHARED_LIB_POM));
    mavenModelBuilder.addSharedLibraryDependency(DERBY_GROUP_ID, DERBY_ARTIFACT_ID);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    List<SharedLibrary> sharedLibraries = parser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is(DERBY_GROUP_ID));
    assertThat(sharedLibrary.getArtifactId(), is(DERBY_ARTIFACT_ID));
  }

  @Test
  public void addDependencyTestCase() {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(getPath(BASIC_POM));
    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .build();
    BundleDependency dependency = new BundleDependency.Builder().setBundleDescriptor(descriptor).build();
    mavenModelBuilder.addDependency(dependency);
    MavenPomParser parser = getPomParserFromMavenModelBuilder(mavenModelBuilder);
    List<BundleDependency> dependencies = parser.getDependencies();
    assertThat(dependencies, hasSize(4));
    assertThat(dependencies, hasItem(dependency));
  }

  @Test
  public void createPropertyFileTestCase() throws IOException {
    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl(GROUP_ID, ARTIFACT_ID, VERSION, empty(), of(PACKAGING));
    Properties properties = new Properties();
    properties.put(PROPERTY_KEY, PROPERTY_VALUE);
    mavenModelBuilder.createDeployablePomProperties(temporaryFolder.getRoot().toPath(), properties);
    File propertyFile = get(temporaryFolder.getRoot().getPath(),
                            "META-INF", "maven", GROUP_ID, ARTIFACT_ID, MULE_POM_PROPERTIES).toFile();
    assertThat(propertyFile, anExistingFile());
    Properties tmpProperties = new Properties();
    try (InputStream resourceStream = new FileInputStream(propertyFile)) {
      tmpProperties.load(resourceStream);
      assertThat(tmpProperties.getProperty(PROPERTY_KEY), is(PROPERTY_VALUE));
    }
  }

  private MavenPomParser getPomParserFromMavenModelBuilder(MavenModelBuilder mavenModelBuilder) {
    Path tmpPath = temporaryFolder.getRoot().toPath();
    mavenModelBuilder.createDeployablePomFile(tmpPath);
    return new MavenPomParserImpl(tmpPath);
  }
}
