/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.util;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.IOUtils.toByteArray;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

public class FileUtils {

  private static final String META_INF = "META-INF";
  private static final String MAVEN_PATH = META_INF + "/maven";
  private static final String MULE_PLUGIN_POM = "pom.xml";

  /**
   * Finds the URL of the pom file within the artifact file.
   *
   * @param artifactFile the artifact file to search for the pom file.
   * @return the URL to the pom file.
   */
  public static URL getPomUrlFromJar(File artifactFile) {
    try {
      List<URL> jarMavenUrls = getUrlsWithinJar(artifactFile, MAVEN_PATH);
      if (jarMavenUrls.isEmpty()) {
        try (JarFile jarFile = new JarFile(artifactFile)) {
          throw new RuntimeException(format("No entries found in %s for artifact %s.\nFound entries: %s",
                                            MAVEN_PATH,
                                            artifactFile.getName(),
                                            jarFile.stream().map(JarEntry::getName).collect(toList())));
        }
      }
      Optional<URL> pomUrl = jarMavenUrls.stream().filter(url -> url.toString().endsWith(MULE_PLUGIN_POM)).findAny();
      if (!pomUrl.isPresent()) {
        String expectedPath = MAVEN_PATH + "/${groupId}/${artifactId}/" + MULE_PLUGIN_POM;
        throw new RuntimeException(format("The file '%s' was missing while looking within the artifact %s (it should be placed under [%s]).\nFound URLs in Maven folder: %s",
                                          MULE_PLUGIN_POM,
                                          artifactFile.getName(),
                                          expectedPath,
                                          jarMavenUrls));
      }
      return pomUrl.get();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Loads the content of a file within a jar into a byte array.
   *
   * @param jarFile the jar file
   * @return the content of the file as byte array or empty if the file does not exists within the jar file.
   * @throws IOException if there was a problem reading from the jar file.
   */
  public static Optional<byte[]> loadFileContentFrom(URL jarFile) throws IOException {
    JarURLConnection jarConnection = (JarURLConnection) jarFile.openConnection();
    jarConnection.setUseCaches(false);

    try (InputStream stream = jarConnection.getInputStream()) {
      return of(toByteArray(stream));
    } catch (FileNotFoundException e) {
      return empty();
    }
  }

  /**
   * Creates an URL to a path within a jar file.
   *
   * @param jarFile  the jar file
   * @param filePath the path within the jar file
   * @return an URL to the {@code filePath} within the {@code jarFile}
   * @throws MalformedURLException if the provided {@code filePath} is malformed
   */
  private static URL getUrlWithinJar(File jarFile, String filePath) throws MalformedURLException {
    return new URL("jar:" + jarFile.toURI().toString() + "!/" + filePath);
  }

  /**
   * Gets all the URL of files within a directory
   *
   * @param file      the jar file
   * @param directory the directory within the jar file (using "/" as path separator).
   * @return a collection of URLs to files within the directory {@code directory}. Empty collection if the directory does not
   *         exists or is empty.
   * @throws IOException
   */
  private static List<URL> getUrlsWithinJar(File file, String directory) throws IOException {
    List<URL> urls = new ArrayList<>();
    try (JarFile jarFile = new JarFile(file)) {
      Enumeration<JarEntry> entries = jarFile.entries();
      while (entries.hasMoreElements()) {
        JarEntry jarEntry = entries.nextElement();
        // https://pkware.cachefly.net/webdocs/casestudies/APPNOTE.TXT - 4.4.17.1
        // The path stored MUST NOT contain a drive or device letter, or a leading slash. All slashes MUST be forward slashes '/'
        // as opposed to backwards slashes '\'
        if (!jarEntry.isDirectory() && jarEntry.getName().startsWith(directory + "/")) {
          urls.add(getUrlWithinJar(file, jarEntry.getName()));
        }
      }
    }
    return urls;
  }

  /**
   * Workaround for JDK bug <a href="http://bugs.sun.com/bugdatabase/view_bug.do;:YfiG?bug_id=4117557"> 4117557</a>. More
   * in-context information at <a href="http://mule.mulesoft.org/jira/browse/MULE-1112">MULE-1112</a>
   * <p/>
   * Factory methods correspond to constructors of the <code>java.io.File class</code>. No physical file created in this method.
   *
   * @see File
   */
  public static File newFile(String pathName) {
    try {
      return new File(pathName).getCanonicalFile();
    } catch (IOException e) {
      throw new RuntimeException("Unable to create a canonical file for " + pathName, e);
    }
  }

}
