/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.model;

import static java.util.Optional.ofNullable;

import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.PomParentCoordinates;

import java.io.File;
import java.util.Optional;

import org.apache.maven.model.Model;

public class MavenPomModelWrapper implements MavenPomModel {

  private final Model model;
  private final String groupId;
  private final String artifactId;
  private final String version;
  private final Optional<String> packaging;
  private final Optional<File> pomFile;

  public MavenPomModelWrapper(Model model) {
    this.model = model;
    this.groupId = model.getGroupId();
    this.artifactId = model.getArtifactId();
    this.version = model.getVersion();
    this.packaging = ofNullable(model.getPackaging());
    this.pomFile = ofNullable(model.getPomFile());
  }

  @Override
  public Optional<PomParentCoordinates> getParent() {
    return ofNullable(model.getParent()).map(PomParentCoordinatesWrapper::new);
  }

  @Override
  public String getGroupId() {
    return this.groupId;
  }

  @Override
  public String getArtifactId() {
    return this.artifactId;
  }

  @Override
  public String getVersion() {
    return this.version;
  }

  @Override
  public String getPackaging() {
    return this.packaging.orElse(null);
  }

  @Override
  public Optional<File> getPomFile() {
    return pomFile;
  }
}
