/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;

import java.util.List;
import java.util.Objects;

/**
 * Describe the additionalDependencies for a plugin configured in the Mule Maven Plugin
 *
 * @since 2.0.0
 */

public class AdditionalPluginDependencies {

  private final String groupId;

  private final String artifactId;

  private final List<BundleDescriptor> additionalDependencies;

  public AdditionalPluginDependencies(String groupId, String artifactId, List<BundleDescriptor> additionalDependencies) {
    requireNonNull(groupId, "groupId cannot be null");
    requireNonNull(artifactId, "artifactId cannot be null");
    this.groupId = groupId;
    this.artifactId = artifactId;
    this.additionalDependencies = unmodifiableList(additionalDependencies);
  }

  public AdditionalPluginDependencies(AdditionalPluginDependencies old, List<BundleDescriptor> additionalDependencies) {
    this(old.getGroupId(), old.getArtifactId(), additionalDependencies);
  }

  /**
   * Gets the groupId of the plugin.
   *
   * @return the groupId
   */
  public String getGroupId() {
    return groupId;
  }

  /**
   * Gets the artifactId of the plugin.
   *
   * @return the artifactId
   */
  public String getArtifactId() {
    return artifactId;
  }

  /**
   * Gets the additionalDependencies of the plugin.
   *
   * @return the additional dependencies
   */
  public List<BundleDescriptor> getAdditionalDependencies() {
    return additionalDependencies;
  }

  @Override
  public String toString() {
    return groupId + ":" + artifactId;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    AdditionalPluginDependencies that = (AdditionalPluginDependencies) o;
    return groupId.equals(that.groupId) && artifactId.equals(that.artifactId);
  }

  @Override
  public int hashCode() {
    return Objects.hash(groupId, artifactId);
  }

}
