/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.pom.parser.internal.model.MavenModelBuilderImpl.getDefaultModelBuilder;

import static java.lang.String.format;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.SystemUtils.IS_OS_LINUX;
import static org.apache.commons.lang3.SystemUtils.IS_OS_MAC;
import static org.apache.commons.lang3.SystemUtils.IS_OS_WINDOWS;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.assertThat;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenModelBuilder;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class SettingsProfilesActivationTestCase extends AbstractMavenClientTestCase {

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private static final String GROUP_ID = "org.mule";
  private static final String ARTIFACT_ID = "mule-core";
  private static final String VERSION = "3.3.0";
  private static final String TYPE = "pom";

  private static final String JDK = "jdk";
  private static final String OS = "os";
  private static final String PROPERTY = "property";
  private static final String FILE = "file";

  private static final String SOME_PROPERTY = "someProperty";

  @Override
  protected void beforeTest() throws Exception {

  }

  private URL getPomUrl(String activation, String settingsFilePath) {
    return getClass().getClassLoader().getResource(format("activation-settings/active-by-%s/%s", activation, settingsFilePath));
  }

  private MavenConfiguration.MavenConfigurationBuilder getMavenConfigurationBuilder(URL settingsUrl) {
    return newMavenConfigurationBuilder().globalSettingsLocation(toFile(settingsUrl));
  }

  private void test(String activation, boolean active, String settingsFile) {
    test(activation, active, settingsFile, empty());
  }

  private void test(String activation, boolean active, String settingsFile, Optional<Properties> userProperties) {
    test(activation, active, settingsFile, userProperties, emptyList());
  }

  private void test(String activation, boolean active, String settingsFile, Optional<Properties> userProperties,
                    List<String> activeProfiles) {
    test(getMavenConfigurationBuilder(getPomUrl(activation, settingsFile)), active, BundleDependenciesResolutionException.class,
         of(ArtifactResolutionException.class), empty(), userProperties, activeProfiles);
  }

  private void test(MavenConfiguration.MavenConfigurationBuilder builder, boolean active, Class expectedError,
                    Optional<Class> expectedErrorCause, Optional<String> errorMessage,
                    Optional<Properties> userProperties, List<String> activeProfiles) {
    if (!active) {
      expectedException.expect(expectedError);
      expectedErrorCause.ifPresent(errorCauseClass -> expectedException.expectCause(instanceOf(errorCauseClass)));
      errorMessage.ifPresent(message -> expectedException.expectMessage(message));
    }
    File localRepository;
    try {
      localRepository = temporaryFolder.newFolder();
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
    userProperties.ifPresent(properties -> builder.userProperties(properties));
    builder.activeProfiles(activeProfiles);
    builder.localMavenRepositoryLocation(localRepository);

    mavenClient = mavenClientProvider.createMavenClient(builder.build());

    BundleDependency resolved = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .setType(TYPE).build());

    BundleDescriptor descriptor = resolved.getDescriptor();
    assertThat(descriptor.getGroupId(), equalTo(GROUP_ID));
    assertThat(descriptor.getArtifactId(), equalTo(ARTIFACT_ID));
    assertThat(descriptor.getVersion(), equalTo(VERSION));
    assertThat(descriptor.getType(), equalTo(TYPE));
  }

  @Test
  public void activeByExistentSystemProperty() {
    String oldValue = setProperty(SOME_PROPERTY, "true");
    try {
      test(PROPERTY, true, "settings.xml");
    } finally {
      if (oldValue == null) {
        clearProperty(SOME_PROPERTY);
      } else {
        setProperty(SOME_PROPERTY, oldValue);
      }
    }
  }

  @Test
  public void activeByExistentUserProperty() {
    Properties properties = new Properties();
    properties.setProperty(SOME_PROPERTY, "true");
    test(PROPERTY, true, "settings.xml", of(properties));
  }

  @Test
  public void activeProfileExplicitly() {
    String profileId = "muleCentralRepo";
    test(PROPERTY, true, "settings.xml", empty(), singletonList(profileId));
  }

  @Test
  public void inactiveByExistentUserProperty() {
    Properties properties = new Properties();
    properties.setProperty(SOME_PROPERTY, "false");
    test(PROPERTY, false, "settings.xml", of(properties));
  }

  @Test
  public void inactiveByNonExistentProperty() {
    test(PROPERTY, false, "settings.xml");
  }

  @Test
  public void fileActivationNotSupported() {
    test(getMavenConfigurationBuilder(getPomUrl(FILE, "settings.xml")), false,
         UnsupportedOperationException.class,
         empty(),
         of("Error while resolving dependencies, profile 'muleCentralRepo' resolved from effective settings has defined a profile activation by file which is not supported"),
         empty(), emptyList());
  }

  @Test
  public void activationByLinuxOS() {
    test(OS, IS_OS_LINUX, "settings-linux.xml");
  }

  @Test
  public void activationByMacOS() {
    test(OS, IS_OS_MAC, "settings-mac.xml");
  }

  @Test
  public void activationByWindowsOS() {
    test(OS, IS_OS_WINDOWS, "settings-windows.xml");
  }

  @Test
  public void activeByCorrectJDK() throws IOException {
    MavenModelBuilder mavenModelBuilder = getDefaultModelBuilder();

    mavenModelBuilder.getNewMavenProfileBuilder()
        .setActiveByDefault(false)
        .setActivationByJdk(getProperty("java.specification.version"))
        .addRepository("maven-central", "MavenCentral", "https://repo.maven.apache.org/maven2/")
        .addRepository("mulesoft-public", "MuleSoft Public",
                       "https://repository.mulesoft.org/nexus/content/repositories/public/")
        .build();

    File pomFile = temporaryFolder.newFile("settings-active.xml");

    mavenModelBuilder.updateArtifactPom(pomFile.toPath());

    test(getMavenConfigurationBuilder(pomFile.toURI().toURL()), true, BundleDependenciesResolutionException.class,
         empty(), empty(), empty(), emptyList());
  }

  @Test
  public void inactiveByOldJDK() {
    test(JDK, false, "settings-inactive.xml");
  }

}
