/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.test.AllureConstants.MavenClient.RemoteRepositories.MERGE_REMOTE_REPOSITORIES;

import static java.util.Collections.singletonList;

import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;

import org.mule.maven.client.internal.RemoteRepositoriesMerger;

import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.eclipse.aether.repository.RemoteRepository;
import org.junit.Test;

@Story(MERGE_REMOTE_REPOSITORIES)
public class RemoteRepositoriesMergerTestCase extends AbstractMavenClientTestCase {

  private static final String MULE_ID = "mule";
  private static final String MULE_PUBLIC_URL = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULE_PRIVATE_URL = "https://repository.mulesoft.org/nexus/content/repositories/private/";
  public static final String MULE_PRIVATE_ID = "mule-private";

  private RemoteRepositoriesMerger remoteRepositoriesMerger;

  private RemoteRepository dominantRemoteRepository;
  private RemoteRepository recessiveRemoteRepository;

  @Override
  protected void beforeTest() throws Exception {
    remoteRepositoriesMerger = new RemoteRepositoriesMerger();

    dominantRemoteRepository = new RemoteRepository.Builder(MULE_ID, null, MULE_PUBLIC_URL).build();
    recessiveRemoteRepository =
        new RemoteRepository.Builder(MULE_PRIVATE_ID, null, MULE_PRIVATE_URL).build();
  }

  @Description("Validates that when two repositories have same id the dominant one has precedence")
  @Test
  public void shouldIncludeDominantWhenSameId() {
    recessiveRemoteRepository = new RemoteRepository.Builder(recessiveRemoteRepository).setId(MULE_ID).build();
    List<RemoteRepository> merge =
        remoteRepositoriesMerger.merge(singletonList(dominantRemoteRepository), singletonList(recessiveRemoteRepository));

    assertThat(merge, hasSize(1));
    final RemoteRepository mergedRemoteRepository = merge.get(0);
    assertThat(mergedRemoteRepository.getId(), is(MULE_ID));
    assertThat(mergedRemoteRepository.getUrl(), is(MULE_PUBLIC_URL));
  }

  @Description("Validates that when two repositories have different ids both are added")
  @Test
  public void shouldIncludeDominantWhenDifferentId() {
    List<RemoteRepository> merge =
        remoteRepositoriesMerger.merge(singletonList(dominantRemoteRepository), singletonList(recessiveRemoteRepository));

    assertThat(merge, hasSize(2));
    assertThat(merge, hasItems(recessiveRemoteRepository, dominantRemoteRepository));
  }

}
