/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import static java.lang.String.format;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.SystemUtils.IS_OS_LINUX;
import static org.apache.commons.lang3.SystemUtils.IS_OS_MAC;
import static org.apache.commons.lang3.SystemUtils.IS_OS_WINDOWS;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.MuleMavenRepositoryState;
import org.mule.maven.client.internal.MuleMavenRepositoryStateFactory;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenModelBuilder;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.internal.MavenPomParserImpl;
import org.mule.maven.pom.parser.internal.model.MavenModelBuilderImpl;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.installation.InstallRequest;
import org.eclipse.aether.installation.InstallationException;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class PomProfilesActivationTestCase extends AbstractMavenClientTestCase {

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private static final String GROUP_ID = "org.mule";
  private static final String ARTIFACT_ID = "mule-core";
  private static final String VERSION = "3.3.0";
  private static final String TYPE = "pom";

  private static final String SOME_PROPERTY = "someProperty";

  private static MuleMavenRepositoryStateFactory muleMavenRepositoryStateFactory;

  private MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder;
  private MuleMavenRepositoryState muleMavenRepositoryState;

  @BeforeClass
  public static void classSetUp() {
    muleMavenRepositoryStateFactory = new MuleMavenRepositoryStateFactory();
  }

  @Override
  protected void beforeTest() throws Exception {
    File localMavenRepository = repositoryFolder.newFolder();

    mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository)
        .ignoreArtifactDescriptorRepositories(false);

    muleMavenRepositoryState =
        muleMavenRepositoryStateFactory.createMavenRepositoryState(localMavenRepository, empty(), empty(), empty(), empty(),
                                                                   emptyMap(), false, false, false, empty(), session -> {
                                                                   });
  }

  @AfterClass
  public static void classCleanUp() throws Exception {
    muleMavenRepositoryStateFactory.close();
  }

  @Test
  public void activateProfileByProperty() {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.userProperties(userProperties).build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void activateProfileByJdk() throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    MavenModelBuilder mavenModelBuilder = new MavenModelBuilderImpl("org.mule.tests", "pom-active-by-property",
                                                                    "1.0.0-SNAPSHOT", of("4.0.0"), of("pom"));
    mavenModelBuilder.addRepository("mulesoft-public", "MuleSoft Public",
                                    "https://repository.mulesoft.org/nexus/content/repositories/public/");

    BundleDescriptor descriptor =
        new BundleDescriptor.Builder().setGroupId("org.mule").setArtifactId("mule-core").setVersion("3.3.0")
            .setType("pom").setExclusions(singletonList(new ArtifactCoordinates("*", "*"))).build();
    BundleDependency bundleDependency = new BundleDependency.Builder().setBundleDescriptor(descriptor).build();

    MavenModelBuilder.MavenProfileBuilder mavenProfileBuilder = mavenModelBuilder.getNewMavenProfileBuilder();
    mavenProfileBuilder.setActiveByDefault(false)
        .setActivationByJdk(System.getProperty("java.specification.version"))
        .addDependency(bundleDependency)
        .build();

    File pomFile = temporaryFolder.newFile("settings-active.xml");
    mavenModelBuilder.updateArtifactPom(pomFile.toPath());
    BundleDescriptor bundleDescriptor = installArtifact(pomFile);

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void activateProfileByLinuxOs() {
    activeProfileByOs("linux", IS_OS_LINUX);
  }

  @Test
  public void activateProfileByWindowsOs() {
    activeProfileByOs("windows", IS_OS_WINDOWS);
  }

  @Test
  public void activateProfileByMacOs() {
    activeProfileByOs("mac", IS_OS_MAC);
  }

  @Test
  public void fileActivationNotSupported() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    expectedException
        .expectMessage("Error while resolving dependencies for org.mule.tests:pom-active-by-file:pom:1.0.0-SNAPSHOT due to profiles activation by file are not supported");
    expectedException.expect(UnsupportedOperationException.class);
    mavenClient
        .resolveArtifactDependencies(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-file/pom.xml")),
                                     false, false, empty(), empty());
    expectedException.reportMissingExceptionWithMessage("Should not be supported to activate profiles by file");
  }

  @Test
  public void inactivateProfileByJdk() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-jdk/inactive-by-jdk-pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfilesExplicitlyAAreNotConsideredWhenResolvingBundleDescriptorDependencies() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(singletonList("mule-core-dependency")).build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfileExplicitlyAreConsideredWhenResolvingArtifactDependencies() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(singletonList("mule-core-dependency")).build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(getClass().getClassLoader()
            .getResource("activation-poms/active-by-property/pom.xml")),
                                                false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(1));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void inactivateProfileExplicitlyAreNotConsideredWhenResolvingBundleDescriptorDependencies() {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .inactiveProfiles(singletonList("mule-core-dependency"))
        .userProperties(userProperties)
        .build());

    BundleDescriptor bundleDescriptor =
        installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(1));
  }

  @Test
  public void inactivateProfileExplicitlyAreConsideredWhenResolvingArtifactDependencies() {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .inactiveProfiles(singletonList("mule-core-dependency"))
        .userProperties(userProperties)
        .build());

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(getClass().getClassLoader()
            .getResource("activation-poms/active-by-property/pom.xml")),
                                                false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(0));
  }

  @Test
  public void activateProfileBySystemProperty() {
    String oldValue = setProperty(SOME_PROPERTY, "true");
    try {
      mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

      BundleDescriptor bundleDescriptor =
          installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/active-by-property/pom.xml")));

      List<BundleDependency> bundleDependencies =
          mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
      assertThat(bundleDependencies, hasSize(1));
      assertDescriptor(bundleDependencies.get(0).getDescriptor());
    } finally {
      if (oldValue != null) {
        setProperty(SOME_PROPERTY, oldValue);
      } else {
        clearProperty(SOME_PROPERTY);
      }
    }
  }

  @Test
  public void activateProfileByPropertyWithParentPom() {
    Properties userProperties = new Properties();
    userProperties.setProperty(SOME_PROPERTY, "true");

    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.userProperties(userProperties).build());

    installArtifact(toFile(getClass().getClassLoader().getResource("activation-poms/parent-pom/parent-pom.xml")));
    BundleDescriptor bundleDescriptor = installArtifact(toFile(getClass().getClassLoader()
        .getResource("activation-poms/active-by-property-with-parent-pom/pom.xml")));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    assertThat(bundleDependencies, hasSize(2));
    assertDescriptor(bundleDependencies.get(0).getDescriptor());
  }

  @Test
  public void noActivationProfile() {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder
        .activeProfiles(singletonList("mule-core-dependency")).build());

    List<BundleDependency> bundleDependencies = mavenClient
        .resolveArtifactDependencies(toFile(getClass().getClassLoader().getResource("pom-with-no-profile-activation/pom.xml")),
                                     false, false, empty(), empty());
    assertThat(bundleDependencies, hasSize(1));
  }

  public void activeProfileByOs(String os, boolean shouldBeActivated) {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfigurationBuilder.build());

    BundleDescriptor bundleDescriptor = installArtifact(toFile(getClass().getClassLoader()
        .getResource(format("activation-poms/active-by-os/active-by-%s-os-pom.xml", os))));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
    if (shouldBeActivated) {
      assertThat(bundleDependencies, hasSize(1));
      assertDescriptor(bundleDependencies.get(0).getDescriptor());
    } else {
      assertThat(bundleDependencies, hasSize(0));
    }
  }

  private BundleDescriptor installArtifact(File artifactFile) {
    MavenPomModel pomModel = new MavenPomParserImpl(artifactFile.toPath()).getModel();
    InstallRequest request = new InstallRequest();
    request.addArtifact(new DefaultArtifact(pomModel.getGroupId(), pomModel.getArtifactId(), null, "pom", pomModel.getVersion(),
                                            emptyMap(), artifactFile));
    try {
      muleMavenRepositoryState.getSystem().install(muleMavenRepositoryState.getSession(), request);
      return new BundleDescriptor.Builder()
          .setGroupId(pomModel.getGroupId())
          .setArtifactId(pomModel.getArtifactId())
          .setVersion(pomModel.getVersion())
          .setType(pomModel.getPackaging())
          .build();
    } catch (InstallationException e) {
      throw new RuntimeException(e);
    }
  }

  private void assertDescriptor(BundleDescriptor descriptor) {
    assertThat(descriptor.getGroupId(), equalTo(GROUP_ID));
    assertThat(descriptor.getArtifactId(), equalTo(ARTIFACT_ID));
    assertThat(descriptor.getVersion(), equalTo(VERSION));
    assertThat(descriptor.getType(), equalTo(TYPE));
  }

}
