/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.OnlineWithReposStory.ONLINE_STORY;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.stream.Stream;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(ONLINE_STORY)
public class OnlineWithRepositoriesTestCase extends AbstractOnlineMavenClientTestCase {

  @Override
  protected MavenConfiguration createMavenConfiguration() throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.getRoot())
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("mulesoft-public")
            .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public/")).build())
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("maven-central")
            .url(new URL("https://repo.maven.apache.org/maven2/")).build())
        .build();
  }

  @Test
  public void resolveArtifactDependenciesFromPomFileInsideJar() throws MalformedURLException {
    BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId("org.mule.connectors")
        .setArtifactId("mule-sockets-connector")
        .setVersion("1.1.1")
        .setType("jar")
        .setClassifier("mule-plugin")
        .build());
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(toFile(bundleDependency.getBundleUri().toURL()),
                                                false,
                                                false,
                                                of(repositoryFolder.getRoot()),
                                                of(temporaryFolder.getRoot()));
    checkNoDuplicatedInstances(bundleDependencies);
    List<String> result = bundleDependencies.stream()
        .filter(dep -> dep.getBundleUri() != null)
        .map(dep -> dep.getDescriptor().getArtifactId())
        .collect(toList());
    assertThat(result, contains("commons-io", "guava", "jsr305", "checker-qual", "error_prone_annotations", "j2objc-annotations",
                                "animal-sniffer-annotations"));
  }

  @Test
  public void resolveArtifactDependenciesDirectly() {
    BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.connectors")
        .setArtifactId("mule-sockets-connector")
        .setVersion("1.1.1")
        .setType("jar")
        .setClassifier("mule-plugin")
        .build();
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(singletonList(bundleDescriptor),
                                                of(repositoryFolder.getRoot()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, hasSize(1));
    assertThat(bundleDependencies.get(0).getDescriptor(), equalTo(bundleDescriptor));

    List<String> result = bundleDependencies.get(0).getTransitiveDependencies().stream()
        .flatMap(transitiveDependency -> concat(Stream.of(transitiveDependency),
                                                transitiveDependency.getTransitiveDependencies().stream()))
        .filter(dep -> dep.getBundleUri() != null)
        .map(dep -> dep.getDescriptor().getArtifactId())
        .collect(toList());
    assertThat(result, contains("commons-io", "guava", "jsr305", "checker-qual", "error_prone_annotations", "j2objc-annotations",
                                "animal-sniffer-annotations"));
  }

  @Test
  public void resolveArtifactDependenciesDirectlyForNonMulePluginDependencies() {
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(asList(new BundleDescriptor.Builder()
            .setGroupId("commons-collections")
            .setArtifactId("commons-collections")
            .setVersion("3.2.1")
            .build(),
                                                       new BundleDescriptor.Builder()
                                                           .setGroupId("commons-io")
                                                           .setArtifactId("commons-io")
                                                           .setVersion("2.6")
                                                           .build()),
                                                of(repositoryFolder.getRoot()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, hasSize(2));

    List<String> result = bundleDependencies.stream()
        .filter(dep -> dep.getBundleUri() != null)
        .map(dep -> dep.getDescriptor().getArtifactId())
        .collect(toList());
    assertThat(result, contains("commons-collections", "commons-io"));
  }

  @Test
  public void resolveArtifactDependenciesDirectlyDuplicatedDependency() {
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(asList(new BundleDescriptor.Builder()
            .setGroupId("commons-collections")
            .setArtifactId("commons-collections")
            .setVersion("3.2.1")
            .build(),
                                                       new BundleDescriptor.Builder()
                                                           .setGroupId("commons-collections")
                                                           .setArtifactId("commons-collections")
                                                           .setVersion("3.2.2")
                                                           .build()),
                                                of(repositoryFolder.getRoot()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, hasSize(1));

    List<String> result = bundleDependencies.stream()
        .filter(dep -> dep.getBundleUri() != null)
        .map(dep -> dep.getDescriptor().getArtifactId() + ":" + dep.getDescriptor().getVersion())
        .collect(toList());
    // Latest definition is the one that Aether will choose for this scenario
    assertThat(result, contains("commons-collections:3.2.2"));
  }

  @Test
  public void resolveArtifactDependenciesDirectlyForNonMulePluginDependenciesConflictingTransitiveDependencies() {
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(asList(new BundleDescriptor.Builder()
            .setGroupId("org.apache.httpcomponents")
            .setArtifactId("httpclient")
            .setVersion("4.5.9")
            .build(),
                                                       new BundleDescriptor.Builder()
                                                           .setGroupId("commons-codec")
                                                           .setArtifactId("commons-codec")
                                                           .setVersion("1.12")
                                                           .build()),
                                                of(repositoryFolder.getRoot()),
                                                empty());
    checkNoDuplicatedInstances(bundleDependencies);
    assertThat(bundleDependencies, hasSize(4));

    List<String> result = bundleDependencies.stream()
        .filter(dep -> dep.getBundleUri() != null)
        .map(dep -> dep.getDescriptor().getArtifactId() + ":" + dep.getDescriptor().getVersion())
        .collect(toList());
    assertThat(result, contains("httpclient:4.5.9", "httpcore:4.4.11", "commons-logging:1.2", "commons-codec:1.12"));
  }

}
