/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.internal.MuleMavenRepositoryStateFactory.MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY;
import static org.mule.maven.client.test.AllureConstants.MavenClient.ConcurrentLocalRepositoryStory.CONCURRENT_LOCAL_REPOSITORY;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(CONCURRENT_LOCAL_REPOSITORY)
public class RepositoryLockingTestCase extends AbstractMavenClientTestCase {

  private static final String DEFAULT_LOCKS_DIR = ".locks";

  @Override
  protected void beforeTest() throws Exception {}

  private MavenConfiguration createMavenConfiguration() throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.getRoot())
        .remoteRepository(newRemoteRepositoryBuilder()
            .id("mulesoft-public")
            .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public/")).build())
        .build();
  }

  @Test
  public void allowConcurrentUsageOfLocalRepository() throws MalformedURLException {
    String oldPropertyValue = setProperty(MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY, "true");

    try {
      mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());

      String groupId = "org.mule";
      String artifactId = "mule-core";
      String version = "3.3.0";
      String type = "pom";
      final BundleDependency bundleDependency =
          mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
              .setArtifactId(artifactId)
              .setVersion(version)
              .setType(type).build());

      assertThat(bundleDependency, notNullValue());
      assertThat(bundleDependency.getDescriptor().getArtifactId(), equalTo(artifactId));

      File lockDir = new File(repositoryFolder.getRoot(), DEFAULT_LOCKS_DIR);
      assertThat(lockDir.exists(), is(true));
    } finally {
      if (oldPropertyValue != null) {
        setProperty(MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY, oldPropertyValue);
      } else {
        clearProperty(MULE_MAVEN_CLIENT_CONCURRENT_LOCAL_REPOSITORY_PROPERTY);
      }
    }
  }

}
