/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;

import static org.eclipse.aether.resolution.ResolutionErrorPolicy.CACHE_DISABLED;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.hamcrest.core.StringContains.containsString;
import static org.junit.Assert.assertThrows;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import java.io.File;
import java.net.MalformedURLException;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Parent;
import org.apache.maven.model.resolution.ModelResolver;
import org.apache.maven.model.resolution.UnresolvableModelException;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.util.repository.SimpleResolutionErrorPolicy;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Just like {@link DefaultModelResolver} was copied from {@link org.apache.maven.repository.internal.DefaultModelResolver}, this
 * test case has been copied from {@link org.apache.maven.repository.internal.DefaultModelResolverTest}.
 */
public class DefaultModelResolverTestCase {

  private static final File LOCAL_REPOSITORY_LOCATION = new File("target/test-classes/repo");
  private static MuleMavenRepositoryStateFactory muleMavenRepositoryStateFactory;
  private static MuleMavenRepositoryState muleMavenRepositoryState;

  private static RemoteRepository getTestRepository() {
    try {
      return new RemoteRepository.Builder(
                                          "repo",
                                          "default",
                                          LOCAL_REPOSITORY_LOCATION.toURI().toURL().toString())
                                              .build();
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

  @BeforeClass
  public static void classSetUp() {
    muleMavenRepositoryStateFactory = new MuleMavenRepositoryStateFactory();
    muleMavenRepositoryState =
        muleMavenRepositoryStateFactory.createMavenRepositoryState(LOCAL_REPOSITORY_LOCATION,
                                                                   empty(),
                                                                   empty(),
                                                                   empty(),
                                                                   empty(),
                                                                   emptyMap(),
                                                                   false,
                                                                   false,
                                                                   false,
                                                                   empty(),
                                                                   session -> session
                                                                       .setResolutionErrorPolicy(new SimpleResolutionErrorPolicy(CACHE_DISABLED)));

  }

  @AfterClass
  public static void classCleanUp() throws Exception {
    muleMavenRepositoryStateFactory.close();
  }

  @Test
  public void resolveParentThrowsUnresolvableModelExceptionWhenNotFound() {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("0");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(parent));

    assertThat(e, hasMessage(containsString("Could not find artifact ut.simple:artifact:pom:0 in repo")));
  }

  @Test
  public void resolveParentThrowsUnresolvableModelExceptionWhenNoMatchingVersionFound() {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("[2.0,2.1)");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(parent));

    assertThat(e, hasMessage(is("No versions matched the requested range '[2.0,2.1)'")));
  }

  @Test
  public void resolveParentThrowsUnresolvableModelExceptionWhenUsingRangesWithoutUpperBound() {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("[1.0,)");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(parent));

    assertThat(e, hasMessage(is("The requested version range '[1.0,)' does not specify an upper bound")));
  }

  @Test
  public void resolveParentSuccessfullyResolvesExistingParentWithoutRange() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("1.0");

    assertThat(createModelResolver().resolveModel(parent), is(notNullValue()));
    assertThat(parent.getVersion(), is("1.0"));
  }

  @Test
  public void resolveParentSuccessfullyResolvesExistingParentUsingHighestVersion() throws Exception {
    final Parent parent = new Parent();
    parent.setGroupId("ut.simple");
    parent.setArtifactId("artifact");
    parent.setVersion("(,2.0)");

    assertThat(createModelResolver().resolveModel(parent), is(notNullValue()));
    assertThat(parent.getVersion(), is("1.0"));
  }

  @Test
  public void resolveDependencyThrowsUnresolvableModelExceptionWhenNotFound() {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("0");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(dependency));

    assertThat(e, hasMessage(containsString("Could not find artifact ut.simple:artifact:pom:0 in repo")));
  }

  @Test
  public void resolveDependencyThrowsUnresolvableModelExceptionWhenNoMatchingVersionFound() {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("[2.0,2.1)");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(dependency));

    assertThat(e, hasMessage(is("No versions matched the requested dependency version range '[2.0,2.1)'")));
  }

  @Test
  public void resolveDependencyThrowsUnresolvableModelExceptionWhenUsingRangesWithoutUpperBound() {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("[1.0,)");

    UnresolvableModelException e = assertThrows(UnresolvableModelException.class,
                                                () -> createModelResolver().resolveModel(dependency));

    assertThat(e, hasMessage(is("The requested dependency version range '[1.0,)' does not specify an upper bound")));
  }

  @Test
  public void resolveDependencySuccessfullyResolvesExistingDependencyWithoutRange() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("1.0");

    assertThat(createModelResolver().resolveModel(dependency), is(notNullValue()));
    assertThat(dependency.getVersion(), is("1.0"));
  }

  @Test
  public void resolveDependencySuccessfullyResolvesExistingDependencyUsingHighestVersion() throws Exception {
    final Dependency dependency = new Dependency();
    dependency.setGroupId("ut.simple");
    dependency.setArtifactId("artifact");
    dependency.setVersion("(,2.0)");

    assertThat(createModelResolver().resolveModel(dependency), is(notNullValue()));
    assertThat(dependency.getVersion(), is("1.0"));
  }

  private ModelResolver createModelResolver() {
    return muleMavenRepositoryState.createModelResolver(singletonList(getTestRepository()));
  }
}
