/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static org.mule.maven.client.internal.MuleMavenClient.MULE_DOMAIN_CLASSIFIER;

import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.Map;

import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.collection.DependencyGraphTransformationContext;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.graph.DependencyNode;

/**
 * Transformer that will go over the graph and look for {@code mule-domain} classified dependencies, once found it will remove all
 * the child dependencies.
 * <p>
 * This way nested {@link DependencyGraphTransformer transformers} will apply Maven's logic and transitive dependencies from
 * {@code mule-domain}s will not interfere.
 *
 * @since 1.0
 */
public class MuleDomainDependencyGraphTransformer implements DependencyGraphTransformer {

  @Override
  public DependencyNode transformGraph(DependencyNode node, DependencyGraphTransformationContext context)
      throws RepositoryException {
    Map<DependencyNode, Object> visitedNodes = new IdentityHashMap<>(512);
    visitGraph(node, visitedNodes, true);
    return node;
  }

  private void visitGraph(DependencyNode node, Map<DependencyNode, Object> visitedNodes, boolean rootNode)
      throws RepositoryException {
    if (!setVisited(node, visitedNodes)) {
      return;
    }
    if (isDomain(node) && !rootNode) {
      removeDependencies(node);
    }

    for (DependencyNode child : node.getChildren()) {
      visitGraph(child, visitedNodes, false);
    }
  }

  /**
   * Marks the specified node as being visited and determines whether the node has been visited before.
   *
   * @param node The node being visited, must not be {@code null}.
   * @return {@code true} if the node has not been visited before, {@code false} if the node was already visited.
   */
  private boolean setVisited(DependencyNode node, Map<DependencyNode, Object> visitedNodes) {
    return visitedNodes.put(node, Boolean.TRUE) == null;
  }

  /**
   * From the node given it will remove any child node that is not a {@code mule-domain}. It goes over the child's of the children
   * nodes.
   *
   * @param node the initial node to start removing nodes that are not {@code mule-plugin} as children.
   */
  private void removeDependencies(DependencyNode node) {
    node.setChildren(new ArrayList<>());
  }

  /**
   * @return {@code true} if node is a {@code mule-domain}.
   */
  private boolean isDomain(DependencyNode node) {
    return node != null && node.getArtifact() != null && MULE_DOMAIN_CLASSIFIER.equals(node.getArtifact().getClassifier());
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    return null != obj && this.getClass().equals(obj.getClass());
  }

  @Override
  public int hashCode() {
    return this.getClass().hashCode();
  }

}
