/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.version;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import org.junit.Test;

public class DefaultVersionUtilsTestCase {

  private final DefaultVersionUtils versionUtils = new DefaultVersionUtils();

  @Test
  public void containsVersionSimple() {
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("1.2.3")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("1.2.3-SNAPSHOT")), is(false));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("1.2.4")), is(false));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("1.3.3")), is(false));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("2.2.3")), is(false));
  }

  @Test
  public void containsVersionRange() {
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("(1.2.0, 1.2.10)")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("[1.2.0, 1.2.3]")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("[1.2.3, 1.2.10]")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("[1.2.3,]")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("[,1.2.3]")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("(1.2.3,)")), is(false));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("(,1.2.3)")), is(false));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("(1.2.3-SNAPSHOT, 1.2.4-SNAPSHOT)")), is(true));
    assertThat(versionUtils.containsVersion("1.2.3", singletonList("(1.2.0-SNAPSHOT, 1.2.3-SNAPSHOT)")), is(false));
  }

  @Test
  public void containsVersionMultiple() {
    // containsVersion is supposed to work in an "any match" fashion, so here goes the OR table:
    // Two bad ones
    assertThat(versionUtils.containsVersion("1.2.3", asList("2.0.0", "1.2.4")), is(false));
    // One bad and one good
    assertThat(versionUtils.containsVersion("1.2.3", asList("2.2.3", "[1.2.3, 1.2.10]")), is(true));
    // One good and one bad
    assertThat(versionUtils.containsVersion("1.2.3", asList("[1.2.0, 1.2.3]", "(1.2.3,)")), is(true));
    // Two good ones
    assertThat(versionUtils.containsVersion("1.2.3", asList("1.2.3", "(1.2.0, 1.2.10)")), is(true));
  }
}
