/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.builder.EqualsBuilder.reflectionEquals;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.internal.MuleMavenResolutionContext.AETHER_CONNECTOR_WAGON_CONFIG;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.MuleMavenResolutionContext;
import org.mule.maven.client.internal.wagon.SimplePlexusWagonConfigurator;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Map;
import java.util.Properties;

import org.apache.maven.wagon.providers.http.HttpWagon;
import org.apache.maven.wagon.shared.http.BasicAuthScope;
import org.apache.maven.wagon.shared.http.HttpConfiguration;
import org.apache.maven.wagon.shared.http.HttpMethodConfiguration;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class SimplePlexusWagonConfiguratorTestCase {

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private Map<String, Object> serverConfigurations;

  @Mock
  private HttpWagon wagon;

  @Captor
  private ArgumentCaptor<HttpConfiguration> httpConfigurationArgumentCaptor;

  private Properties httpHeaders;

  private Properties params;

  @Before
  public void before() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-server-wagon-configuration.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder()
            .id("global-wagon-standards")
            .url(new URL("http://maven.apache.org"))
            .build())

        .build();
    MuleMavenResolutionContext context = new MuleMavenResolutionContext(mavenConfiguration);
    serverConfigurations = context.getServerConfigurations();

    httpHeaders = new Properties();
    httpHeaders.put("Cache-control", "no-cache");
    httpHeaders.put("Cache-store", "no-store");
    httpHeaders.put("Pragma", "no-cache");
    httpHeaders.put("Expires", "0");
    httpHeaders.put("Accept-Encoding", "*");

    params = new Properties();
    params.put("http.authentication.preemptive", "%b,true");
    params.put("http.protocol.cookie-policy", "ignore");
  }

  @After
  public void after() {
    verifyNoMoreInteractions(wagon);
  }

  @Test
  public void globalWagonStandardsProperties() throws Exception {
    SimplePlexusWagonConfigurator configurator = new SimplePlexusWagonConfigurator();
    configurator.configure(wagon, serverConfigurations.get(AETHER_CONNECTOR_WAGON_CONFIG + "global-wagon-standards"));

    Properties httpHeaders = new Properties();
    httpHeaders.put("Authorization", "Basic {base64 encoded string}");
    verify(wagon).setHttpHeaders(httpHeaders);

    BasicAuthScope basicAuthScope = new BasicAuthScope();
    basicAuthScope.setHost("ANY");
    basicAuthScope.setPort("ANY");
    basicAuthScope.setRealm("OAM 11g");
    verify(wagon).setBasicAuthScope(argThat(scope -> reflectionEquals(scope, basicAuthScope)));
  }

  @Test
  public void explicitHttpMethodsConfiguration() throws Exception {
    SimplePlexusWagonConfigurator configurator = new SimplePlexusWagonConfigurator();

    configurator.configure(wagon, serverConfigurations.get(AETHER_CONNECTOR_WAGON_CONFIG + "explicit-http-methods"));
    verify(wagon).setHttpConfiguration(httpConfigurationArgumentCaptor.capture());

    HttpConfiguration httpConfiguration = httpConfigurationArgumentCaptor.getValue();
    assertNotNull(httpConfiguration);

    HttpMethodConfiguration put = httpConfiguration.getPut();
    assertNotNull(put);
    assertThat(put.getHeaders(), equalTo(httpHeaders));
    assertThat(put.getParams().size(), is(0));
    assertThat(put.isUseDefaultHeaders(), equalTo(false));
    assertThat(put.isUsePreemptive(), equalTo(false));

    HttpMethodConfiguration get = httpConfiguration.getGet();
    assertNotNull(get);
    assertThat(get.getReadTimeout(), equalTo(120000));
    assertThat(get.getConnectionTimeout(), equalTo(10000));
    assertThat(get.isUseDefaultHeaders(), equalTo(false));
    assertThat(get.isUsePreemptive(), equalTo(true));
    assertThat(get.getHeaders().size(), is(0));
    assertThat(get.getParams().size(), is(0));

    HttpMethodConfiguration head = httpConfiguration.getHead();
    assertNotNull(head);
    assertThat(head.getParams(), equalTo(params));
    assertThat(head.getHeaders().size(), is(0));
  }

  @Test
  public void allHttpMethodsConfiguration() throws Exception {
    SimplePlexusWagonConfigurator configurator = new SimplePlexusWagonConfigurator();

    configurator.configure(wagon, serverConfigurations.get(AETHER_CONNECTOR_WAGON_CONFIG + "all"));
    verify(wagon).setHttpConfiguration(httpConfigurationArgumentCaptor.capture());

    HttpConfiguration httpConfiguration = httpConfigurationArgumentCaptor.getValue();
    assertNotNull(httpConfiguration);

    HttpMethodConfiguration all = httpConfiguration.getAll();
    assertNotNull(all);
    assertThat(all.getHeaders(), equalTo(httpHeaders));
    assertThat(all.getReadTimeout(), equalTo(120000));
    assertThat(all.getConnectionTimeout(), equalTo(10000));
    assertThat(all.isUseDefaultHeaders(), equalTo(false));
    assertThat(all.isUsePreemptive(), equalTo(true));
    assertThat(all.getParams(), equalTo(params));
  }

}
