/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static org.mule.maven.client.internal.BundleDependencyHelper.artifactToBundleDependency;
import static org.mule.maven.client.internal.MulePluginDependencyGraphTransformer.PLUGIN_ANCESTOR_KEY;
import static org.mule.maven.client.internal.util.MavenModelUtils.convertModelRepository;
import static org.mule.maven.pom.parser.internal.util.FileUtils.getPomUrlFromJar;

import static java.lang.String.format;
import static java.lang.System.getProperties;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.apache.maven.artifact.versioning.VersionRange.createFromVersionSpec;
import static org.apache.maven.model.building.ModelBuildingRequest.VALIDATION_LEVEL_MINIMAL;
import static org.eclipse.aether.artifact.ArtifactProperties.LOCAL_PATH;
import static org.eclipse.aether.util.artifact.ArtifactIdUtils.toId;
import static org.eclipse.aether.util.artifact.JavaScopes.COMPILE;
import static org.eclipse.aether.util.artifact.JavaScopes.PROVIDED;
import static org.eclipse.aether.util.artifact.JavaScopes.SYSTEM;
import static org.eclipse.aether.util.artifact.JavaScopes.TEST;
import static org.eclipse.aether.util.graph.transformer.ConflictResolver.NODE_DATA_WINNER;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.client.api.PomFileSupplierFactory;
import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.exception.BundleDependencyNotFoundException;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.pom.UrlModelSource;
import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.internal.model.MavenPomModelWrapper;
import org.mule.maven.pom.parser.internal.MavenPomParserImpl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import org.apache.maven.artifact.versioning.InvalidVersionSpecificationException;
import org.apache.maven.model.Model;
import org.apache.maven.model.Profile;
import org.apache.maven.model.building.DefaultModelBuilderFactory;
import org.apache.maven.model.building.DefaultModelBuildingRequest;
import org.apache.maven.model.building.FileModelSource;
import org.apache.maven.model.building.ModelBuildingRequest;
import org.apache.maven.model.building.ModelSource2;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.ArtifactType;
import org.eclipse.aether.artifact.ArtifactTypeRegistry;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.artifact.DefaultArtifactType;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.graph.DependencyFilter;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.graph.Exclusion;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.WorkspaceReader;
import org.eclipse.aether.repository.WorkspaceRepository;
import org.eclipse.aether.resolution.ArtifactDescriptorRequest;
import org.eclipse.aether.resolution.ArtifactDescriptorResult;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.ArtifactResult;
import org.eclipse.aether.resolution.DependencyRequest;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.eclipse.aether.resolution.DependencyResult;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.transfer.ArtifactNotFoundException;
import org.eclipse.aether.util.filter.AndDependencyFilter;
import org.eclipse.aether.util.filter.PatternInclusionsDependencyFilter;
import org.eclipse.aether.util.filter.ScopeDependencyFilter;
import org.eclipse.aether.util.graph.visitor.PathRecordingDependencyVisitor;
import org.eclipse.aether.util.graph.visitor.PreorderNodeListGenerator;
import org.slf4j.Logger;

public class MuleMavenClient implements MavenClient {

  public static final String MULE_PLUGIN_CLASSIFIER = "mule-plugin";
  public static final String MULE_DOMAIN_CLASSIFIER = "mule-domain";

  private static final Logger LOGGER = getLogger(MuleMavenClient.class);
  private static final String POM = "pom";
  private static final String JAR = "jar";
  private static final String ANY = "*";

  private final PomFileSupplierFactory pomFileSupplierFactory = new DefaultPomFileSupplierFactory();
  private final MavenConfiguration mavenConfiguration;
  private final MuleMavenResolutionContext muleMavenResolutionContext;
  private final MuleMavenRepositoryStateFactory muleMavenRepositoryStateFactory;

  private Optional<Consumer<DefaultRepositorySystemSession>> sessionConfigurator = empty();

  public MuleMavenClient(MavenConfiguration mavenConfiguration) {
    this.mavenConfiguration = mavenConfiguration;
    this.muleMavenResolutionContext = new MuleMavenResolutionContext(mavenConfiguration);
    this.muleMavenRepositoryStateFactory = new MuleMavenRepositoryStateFactory();
  }

  private MuleMavenRepositoryState getRepositoryState(File localRepositoryLocation,
                                                      Optional<WorkspaceReader> workspaceReader) {
    return muleMavenRepositoryStateFactory.createMavenRepositoryState(localRepositoryLocation, muleMavenResolutionContext,
                                                                      mavenConfiguration, sessionConfigurator, workspaceReader);
  }

  @Override
  public MavenConfiguration getMavenConfiguration() {
    return this.mavenConfiguration;
  }

  public void setSessionConfigurator(Consumer<DefaultRepositorySystemSession> sessionConfigurator) {
    this.sessionConfigurator = of(sessionConfigurator);
  }

  @Override
  public List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                            boolean includeTestDependencies,
                                                            boolean includeProvidedDependencies,
                                                            Optional<File> localRepositoryLocationSupplier,
                                                            Optional<File> temporaryFolder) {
    return resolveArtifactDependencies(artifactFile, includeTestDependencies, includeProvidedDependencies,
                                       localRepositoryLocationSupplier, empty(), temporaryFolder);
  }

  @Override
  public List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                            boolean includeTestDependencies,
                                                            boolean includeProvidedDependencies,
                                                            Optional<File> localRepositoryLocationSupplier,
                                                            Optional<MavenReactorResolver> mavenReactorResolver,
                                                            Optional<File> temporaryFolder) {
    requireNonNull(artifactFile, "artifactFile cannot be null");
    try {
      File localRepositoryLocation =
          localRepositoryLocationSupplier.orElse(this.muleMavenResolutionContext.getLocalRepositoryLocation());

      MuleMavenRepositoryState repositoryState = getRepositoryState(localRepositoryLocation, empty());

      Model model = getMavenEffectiveModel(artifactFile, temporaryFolder, of(repositoryState));

      if (model.getProfiles().stream().map(Profile::getActivation).filter(Objects::nonNull)
          .anyMatch(activation -> activation.getFile() != null)) {
        throw new UnsupportedOperationException(format("Error while resolving dependencies for %s due to profiles activation by file are not supported",
                                                       model.getId()));
      }

      Artifact artifact = new DefaultArtifact(model.getGroupId(), model.getArtifactId(),
                                              null,
                                              POM,
                                              model.getVersion());

      mavenReactorResolver
          .ifPresent(reactor -> repositoryState.getSession().setWorkspaceReader(new MavenWorkspaceReaderAdapter(reactor)));

      List<Dependency> dependencies =
          model.getDependencies().stream()
              .map(modelDependency -> convertModelDependency(modelDependency,
                                                             repositoryState.getSession().getArtifactTypeRegistry()))
              .collect(toList());
      List<Dependency> managedDependencies =
          model.getDependencyManagement() == null ? emptyList()
              : model.getDependencyManagement().getDependencies().stream()
                  .map(modelDependency -> convertModelDependency(modelDependency,
                                                                 repositoryState.getSession().getArtifactTypeRegistry()))
                  .collect(toList());
      List<RemoteRepository> remoteRepositories;
      if (!mavenConfiguration.getIgnoreArtifactDescriptorRepositories()) {
        remoteRepositories =
            model.getRepositories() == null ? emptyList()
                : model.getRepositories().stream()
                    .map(modelRepository -> convertModelRepository(modelRepository))
                    .collect(toList());

        remoteRepositories =
            new RemoteRepositoryModelResolver(repositoryState.getRemoteRepositoryManager(), repositoryState.getSession())
                .resolveRepositories(muleMavenResolutionContext.getRemoteRepositoriesWithoutSuperPomModelRepositories(),
                                     remoteRepositories);
      } else {
        remoteRepositories = muleMavenResolutionContext.getRemoteRepositories();
      }

      if (LOGGER.isInfoEnabled()) {
        LOGGER.info("About to fetch required dependencies for artifact: {}. This may take a while...", toId(artifact));
      }

      return resolveDependencies(dependencies,
                                 managedDependencies,
                                 remoteRepositories,
                                 includeTestDependencies,
                                 includeProvidedDependencies,
                                 repositoryState);
    } catch (DependencyResolutionException e) {
      DependencyNode node = e.getResult().getRoot();
      logUnresolvedArtifacts(node, e);
      throw new RuntimeException(format("There was an issue solving the dependencies for the artifact [%s]",
                                        artifactFile.getAbsolutePath()),
                                 e);
    }
  }

  public List<BundleDependency> resolveArtifactDependencies(List<BundleDescriptor> dependencies,
                                                            Optional<File> localRepositoryLocationSupplier,
                                                            Optional<MavenReactorResolver> mavenReactorResolver) {
    try {
      File localRepositoryLocation =
          localRepositoryLocationSupplier.orElse(this.muleMavenResolutionContext.getLocalRepositoryLocation());

      MuleMavenRepositoryState repositoryState =
          getRepositoryState(localRepositoryLocation, mavenReactorResolver.map(MavenWorkspaceReaderAdapter::new));

      List<Dependency> dependenciesToResolve = dependencies.stream()
          .map(bundleDescriptor -> new Dependency(new DefaultArtifact(
                                                                      bundleDescriptor.getGroupId(),
                                                                      bundleDescriptor.getArtifactId(),
                                                                      bundleDescriptor.getClassifier().orElse(null),
                                                                      bundleDescriptor.getType(),
                                                                      bundleDescriptor.getVersion()),
                                                  null))
          .collect(toList());

      return resolveDependencies(dependenciesToResolve,
                                 emptyList(),
                                 muleMavenResolutionContext.getRemoteRepositories(),
                                 false,
                                 false,
                                 repositoryState);
    } catch (DependencyResolutionException e) {
      DependencyNode node = e.getResult().getRoot();
      logUnresolvedArtifacts(node, e);
      throw new RuntimeException("There was an issue solving the dependencies", e);
    }
  }

  public MavenPomModel getEffectiveModel(File artifactFile, Optional<File> temporaryFolder,
                                         Optional<MuleMavenRepositoryState> repositoryStateOptional) {
    // model.getRepositories() will contain any remote repository declared in pom.xml including the ones from parent pom hierarchy
    // and central (which is hardcoded)
    return new MavenPomModelWrapper(getMavenEffectiveModel(artifactFile, temporaryFolder, repositoryStateOptional));
  }

  @Override
  public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                    BundleDescriptor bundleDescriptor) {
    return resolveBundleDescriptorDependencies(includeTestDependencies, false, bundleDescriptor);
  }

  @Override
  public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                    boolean includeProvidedDependencies,
                                                                    BundleDescriptor bundleDescriptor) {
    try {
      MuleMavenRepositoryState repositoryState =
          getRepositoryState(this.muleMavenResolutionContext.getLocalRepositoryLocation(), empty());
      Artifact artifact = createArtifactFromBundleDescriptor(bundleDescriptor);
      ArtifactDescriptorResult artifactDescriptorResult =
          repositoryState.getSystem().readArtifactDescriptor(repositoryState.getSession(),
                                                             new ArtifactDescriptorRequest(artifact,
                                                                                           muleMavenResolutionContext
                                                                                               .getRemoteRepositories(),
                                                                                           null));
      List<RemoteRepository> remoteRepositories;
      if (mavenConfiguration.getIgnoreArtifactDescriptorRepositories()) {
        remoteRepositories = muleMavenResolutionContext.getRemoteRepositories();
      } else {
        remoteRepositories =
            new RemoteRepositoryModelResolver(repositoryState.getRemoteRepositoryManager(), repositoryState.getSession())
                .resolveRepositories(muleMavenResolutionContext.getRemoteRepositoriesWithoutSuperPomModelRepositories(),
                                     artifactDescriptorResult.getRepositories());
      }

      return resolveDependencies(artifactDescriptorResult.getDependencies(),
                                 artifactDescriptorResult.getManagedDependencies(),
                                 remoteRepositories,
                                 includeTestDependencies,
                                 includeProvidedDependencies,
                                 repositoryState);
    } catch (RepositoryException e) {
      if (isArtifactNotFound(e.getCause())) {
        throw new BundleDependencyNotFoundException(e);
      } else {
        throw new BundleDependenciesResolutionException(e);
      }
    }
  }

  private Artifact createArtifactFromBundleDescriptor(BundleDescriptor bundleDescriptor) {
    return new DefaultArtifact(bundleDescriptor.getGroupId(), bundleDescriptor.getArtifactId(),
                               bundleDescriptor.getClassifier().orElse(null),
                               bundleDescriptor.getType(),
                               bundleDescriptor.getVersion());
  }

  @Override
  public BundleDependency resolveBundleDescriptor(BundleDescriptor bundleDescriptor) {
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Resolving artifact with resolution context: " + muleMavenResolutionContext.toString());
      }

      MuleMavenRepositoryState repositoryState =
          getRepositoryState(mavenConfiguration.getLocalMavenRepositoryLocation(), empty());

      Artifact artifact = createArtifactFromBundleDescriptor(bundleDescriptor);
      ArtifactRequest artifactRequest =
          new ArtifactRequest(artifact, muleMavenResolutionContext.getRemoteRepositories(), null);
      ArtifactResult artifactResult =
          repositoryState.getSystem().resolveArtifact(repositoryState.getSession(), artifactRequest);
      Artifact resolvedArtifact = artifactResult.getArtifact();
      return artifactToBundleDependency(resolvedArtifact, "compile").build();
    } catch (ArtifactResolutionException e) {
      if (isArtifactNotFound(e.getCause())) {
        throw new BundleDependencyNotFoundException(e);
      } else {
        throw new BundleDependenciesResolutionException(e);
      }
    }
  }

  private boolean isArtifactNotFound(Throwable e) {
    return e instanceof ArtifactNotFoundException;
  }

  @Override
  public MavenPomModel getRawPomModel(File artifactFile) {
    return new MavenPomParserImpl(artifactFile.toPath()).getModel();
  }

  @Override
  public MavenPomModel getEffectiveModel(File artifactFile, Optional<File> temporaryFolder) {
    return this.getEffectiveModel(artifactFile, temporaryFolder, empty());
  }

  @Override
  public List<BundleDependency> resolvePluginBundleDescriptorsDependencies(List<BundleDescriptor> bundleDescriptors) {
    MuleMavenRepositoryState repositoryState =
        getRepositoryState(mavenConfiguration.getLocalMavenRepositoryLocation(), empty());

    try {
      CollectRequest collectRequest = new CollectRequest();
      collectRequest.setRepositories(muleMavenResolutionContext.getRemoteRepositories());
      collectRequest.setDependencies(bundleDescriptors.stream().map(bundleDescriptor -> {
        // This method is meant to resolve mule-plugin only
        return new Dependency(new DefaultArtifact(bundleDescriptor.getGroupId(), bundleDescriptor.getArtifactId(),
                                                  MULE_PLUGIN_CLASSIFIER, "jar", bundleDescriptor.getVersion()),
                              COMPILE);
      }).collect(toList()));
      return doResolveDependencies(repositoryState, collectRequest);
    } catch (DependencyResolutionException e) {
      DependencyNode node = e.getResult().getRoot();
      logUnresolvedArtifacts(node, e);
      throw new RuntimeException(format("There was an issue solving the dependencies for the bundleDescriptors [%s]",
                                        bundleDescriptors),
                                 e);
    } catch (RepositoryException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public VersionRangeResult resolveVersionRange(BundleDescriptor bundleDescriptor) {
    MuleMavenRepositoryState repositoryState =
        getRepositoryState(this.muleMavenResolutionContext.getLocalRepositoryLocation(), empty());
    try {
      createFromVersionSpec(bundleDescriptor.getVersion());
      final org.eclipse.aether.resolution.VersionRangeResult versionRangeResult = repositoryState
          .getSystem().resolveVersionRange(repositoryState.getSession(),
                                           new VersionRangeRequest(createArtifactFromBundleDescriptor(bundleDescriptor),
                                                                   muleMavenResolutionContext
                                                                       .getRemoteRepositories(),
                                                                   null));
      return new DefaultVersionRangeResult(versionRangeResult);
    } catch (InvalidVersionSpecificationException | VersionRangeResolutionException e) {
      throw new RuntimeException(format("Couldn't resolve version range for bundleDescriptor: '%s'",
                                        bundleDescriptor.toString(), e));
    }
  }

  private List<BundleDependency> resolveDependencies(List<Dependency> dependencies,
                                                     List<Dependency> managedDependencies,
                                                     List<RemoteRepository> remoteRepositories,
                                                     boolean enableTestDependencies,
                                                     boolean enableProvidedDependencies,
                                                     MuleMavenRepositoryState repositoryState)
      throws DependencyResolutionException {
    final CollectRequest collectRequest = new CollectRequest();

    if (!dependencies.isEmpty()) {
      collectRequest.setDependencies(dependencies);
    }
    if (!managedDependencies.isEmpty()) {
      collectRequest.setManagedDependencies(managedDependencies);
    }
    if (!remoteRepositories.isEmpty()) {
      collectRequest.setRepositories(remoteRepositories);
    }

    return doResolveDependencies(repositoryState, collectRequest, enableTestDependencies, enableProvidedDependencies);
  }

  private List<BundleDependency> doResolveDependencies(MuleMavenRepositoryState repositoryState,
                                                       CollectRequest collectRequest)
      throws DependencyResolutionException {
    return doResolveDependencies(repositoryState, collectRequest, false, false);
  }

  private List<BundleDependency> doResolveDependencies(MuleMavenRepositoryState repositoryState,
                                                       CollectRequest collectRequest,
                                                       boolean enableTestDependencies,
                                                       boolean enableProvidedDependencies)
      throws DependencyResolutionException {

    // collect has to include all the scopes in order to solve cases like this one:
    //
    // Declared dependencies are:
    // artifact-a -> artifact-b -> artifact-c:1.0-SNAPSHOT
    // test-artifact-a -> artifact-c:2.0-SNAPSHOT
    //
    // artifact-c:2.0-SNAPSHOT wins even if it's under a dependency with TEST scope,
    // as shown in the dependency tree:
    //
    // [INFO] test:app:pom:1.0-SNAPSHOT
    // [INFO] +- test:artifact-a:pom:1.0-SNAPSHOT:compile
    // [INFO] | \- test:artifact-b:pom:1.0-SNAPSHOT:compile
    // [INFO] | \- test:artifact-c:pom:2.0-SNAPSHOT:compile
    // [INFO] \- test:test-artifact-a:pom:1.0-SNAPSHOT:test
    // [INFO] \- (test:artifact-c:pom:2.0-SNAPSHOT:compile - scope updated from test; omitted for duplicate)

    // mule-domains will need to be excluded from collect scope as those pom.xml may not be present in local neither
    // remote repositories when deploying an application. This can happen because the domain was previously deployed in
    // the Runtime but was never deployed to a remote repository
    Set<Dependency> domainDependencies = collectRequest.getDependencies().stream()
        .filter(dependency -> MULE_DOMAIN_CLASSIFIER.equals(dependency.getArtifact().getClassifier())
            && dependency.getScope().equalsIgnoreCase(PROVIDED))
        .collect(toSet());
    collectRequest.setDependencies(collectRequest.getDependencies().stream()
        .filter(dependency -> !domainDependencies.contains(dependency)).collect(toList()));

    List<String> resolutionScopesToExclude = new ArrayList<>(2);

    // by default the resolution scope is set as RUNTIME_PLUS_SYSTEM (compile + system + runtime dependencies), provided
    // and test scopes are excluded from resolution, collection phase includes all
    resolutionScopesToExclude.add(PROVIDED);
    resolutionScopesToExclude.add(TEST);

    // test scope will exclude only provided scope
    if (enableTestDependencies) {
      resolutionScopesToExclude.remove(TEST);
    }
    DependencyFilter resolutionScopeDependencyFilter = new ScopeDependencyFilter(null, resolutionScopesToExclude);

    final DependencyRequest dependencyRequest = new DependencyRequest();
    dependencyRequest.setCollectRequest(collectRequest);
    dependencyRequest.setFilter(new AndDependencyFilter(resolutionScopeDependencyFilter,
                                                        (node, parents) -> !node.getData().containsKey(NODE_DATA_WINNER)));

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Collecting and resolving transitive dependencies for request: {}", collectRequest);
    }
    final DependencyResult dependencyResult =
        repositoryState.getSystem().resolveDependencies(repositoryState.getSession(), dependencyRequest);

    DependencyNode root = dependencyResult.getRoot();

    PreorderNodeListGenerator nlg = new PreorderNodeListGenerator();
    root.accept(nlg);

    BundleDependencyHelper bundleDependencyHelper = new BundleDependencyHelper();
    List<String> scopesToFilterFromResult = new ArrayList<>(resolutionScopesToExclude);
    if (enableProvidedDependencies && !enableTestDependencies) {
      scopesToFilterFromResult.clear();
      scopesToFilterFromResult.add(TEST);
    } else if (enableTestDependencies && enableProvidedDependencies) {
      scopesToFilterFromResult.clear();
    }

    DependencyFilter bundleDependenciesFilter = new ScopeDependencyFilter(null, scopesToFilterFromResult);
    DependencyFilter childDependenciesFilter =
        new AndDependencyFilter(new ScopeDependencyFilter(null, PROVIDED), bundleDependenciesFilter);
    List<BundleDependency> bundleDependencies = nlg.getNodes()
        .stream()
        .filter(node -> !node.getData().containsKey(NODE_DATA_WINNER))
        .filter(node -> !node.getData().containsKey(PLUGIN_ANCESTOR_KEY) || node.getData().get(PLUGIN_ANCESTOR_KEY).equals(root))
        .filter(node -> bundleDependenciesFilter.accept(node, emptyList()))
        .map(node -> bundleDependencyHelper.getBundleDependency(node, childDependenciesFilter))
        .collect(toList());

    // finally we have to explicitly add mule-domain dependency if there was one in the effective list of bundleDependencies to be
    // resolved
    if (enableProvidedDependencies) {
      bundleDependencies.addAll(domainDependencies.stream()
          .map(dependency -> artifactToBundleDependency(dependency.getArtifact(), dependency.getScope()).build())
          .collect(toList()));
    }

    return bundleDependencies;
  }

  private static Dependency convertModelDependency(org.apache.maven.model.Dependency modelDependency,
                                                   ArtifactTypeRegistry artifactTypeRegistry) {
    ArtifactType artifactType = artifactTypeRegistry.get(modelDependency.getType());
    if (artifactType == null) {
      artifactType =
          new DefaultArtifactType(modelDependency.getType(), modelDependency.getType(), modelDependency.getClassifier(), "none");
    }
    Artifact artifactDependency = new DefaultArtifact(
                                                      modelDependency.getGroupId(),
                                                      modelDependency.getArtifactId(),
                                                      modelDependency.getClassifier() != null ? modelDependency.getClassifier()
                                                          : artifactType.getClassifier(),
                                                      artifactType.getExtension(),
                                                      modelDependency.getVersion(),
                                                      SYSTEM.equals(modelDependency.getScope())
                                                          ? singletonMap(LOCAL_PATH, modelDependency.getSystemPath())
                                                          : emptyMap(),
                                                      SYSTEM.equals(modelDependency.getScope())
                                                          ? new File(modelDependency.getSystemPath())
                                                          : null);
    Dependency dependency =
        new Dependency(artifactDependency, modelDependency.getScope());
    // Just to make sure we don't loose this information but optionality will not affect the resolution at this point
    dependency = dependency.setOptional(modelDependency.isOptional());
    dependency = dependency.setExclusions(modelDependency.getExclusions()
        .stream().map(modelExclusion -> new Exclusion(modelExclusion.getGroupId(), modelExclusion.getArtifactId(),
                                                      ANY, ANY))
        .collect(toList()));
    return dependency;
  }

  private void setPom(ModelBuildingRequest modelBuildingRequest, File artifactFile, Optional<File> temporaryFolder) {
    ModelSource2 modelSource;
    if (artifactFile.isFile() && !getExtension(artifactFile.getName()).equalsIgnoreCase(JAR)) {
      modelSource = new FileModelSource(artifactFile);
    } else {
      MavenPomParser parser = new MavenPomParserImpl(artifactFile.toPath());
      MavenPomModel rawPomModel = parser.getModel();
      BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
          .setGroupId(rawPomModel.getGroupId() != null ? rawPomModel.getGroupId() : rawPomModel.getParent().get().getGroupId())
          .setArtifactId(rawPomModel.getArtifactId())
          .setVersion(rawPomModel.getVersion() != null ? rawPomModel.getVersion() : rawPomModel.getParent().get().getVersion())
          .build();

      if (artifactFile.isDirectory()) {
        modelSource =
            new FileModelSource(pomFileSupplierFactory.uncompressPomArtifactSupplier(artifactFile, bundleDescriptor).get());
      } else {
        modelSource = temporaryFolder
            .<ModelSource2>map(
                               t -> new FileModelSource(pomFileSupplierFactory
                                   .compressedArtifactSupplier(artifactFile, bundleDescriptor, t).get()))
            .orElse(new UrlModelSource(getPomUrlFromJar(artifactFile)));
      }
    }
    if (modelSource instanceof FileModelSource) {
      modelBuildingRequest.setPomFile(((FileModelSource) modelSource).getFile());
    }
    modelBuildingRequest.setModelSource(modelSource);
  }

  private void logUnresolvedArtifacts(DependencyNode node, DependencyResolutionException e) {
    List<ArtifactResult> artifactResults = e.getResult().getArtifactResults().stream()
        .filter(artifactResult -> !artifactResult.getExceptions().isEmpty()).collect(toList());

    final List<String> patternInclusion =
        artifactResults.stream().map(artifactResult -> toId(artifactResult.getRequest().getArtifact()))
            .collect(toList());

    PathRecordingDependencyVisitor visitor =
        new PathRecordingDependencyVisitor(new PatternInclusionsDependencyFilter(patternInclusion),
                                           node.getArtifact() != null);
    node.accept(visitor);

    visitor.getPaths().stream().forEach(path -> {
      List<DependencyNode> unresolvedArtifactPath =
          path.stream().filter(dependencyNode -> dependencyNode.getArtifact() != null).collect(toList());
      if (!unresolvedArtifactPath.isEmpty()) {
        if (LOGGER.isWarnEnabled()) {
          LOGGER.warn("Dependency path to not resolved artifacts -> " + unresolvedArtifactPath);
        }
      }
    });
  }

  @Override
  public void close() throws Exception {
    muleMavenRepositoryStateFactory.close();
  }

  /**
   * Custom implementation of a {@link WorkspaceReader} meant to be adapt {@link MavenReactorResolver} to {@link WorkspaceReader}.
   *
   * @since 1.1
   */
  private static class MavenWorkspaceReaderAdapter implements WorkspaceReader {

    private final WorkspaceRepository workspaceRepository;
    private final MavenReactorResolver mavenReactorResolver;


    MavenWorkspaceReaderAdapter(MavenReactorResolver mavenReactorResolver) {
      this.workspaceRepository = new WorkspaceRepository(format("worskpace-repository-reactor"));
      this.mavenReactorResolver = mavenReactorResolver;
    }

    @Override
    public WorkspaceRepository getRepository() {
      return workspaceRepository;
    }

    @Override
    public File findArtifact(Artifact artifact) {
      return mavenReactorResolver.findArtifact(toBundleDescriptor(artifact));
    }

    public BundleDescriptor toBundleDescriptor(Artifact artifact) {
      return new BundleDescriptor.Builder()
          .setGroupId(artifact.getGroupId())
          .setArtifactId(artifact.getArtifactId())
          .setVersion(artifact.getVersion())
          .setBaseVersion(artifact.getBaseVersion())
          .setType(artifact.getExtension())
          .build();
    }

    @Override
    public List<String> findVersions(Artifact artifact) {
      return mavenReactorResolver.findVersions(toBundleDescriptor(artifact));
    }

  }

  private Model getMavenEffectiveModel(File artifactFile, Optional<File> temporaryFolder,
                                       Optional<MuleMavenRepositoryState> repositoryStateOptional) {
    Model model;
    try {
      MuleMavenRepositoryState repositoryState = repositoryStateOptional
          .orElseGet(() -> getRepositoryState(this.muleMavenResolutionContext.getLocalRepositoryLocation(), empty()));
      DefaultModelBuildingRequest defaultModelBuildingRequest = new DefaultModelBuildingRequest();
      setPom(defaultModelBuildingRequest, artifactFile, temporaryFolder);
      mavenConfiguration.getUserProperties().ifPresent(properties -> defaultModelBuildingRequest.setUserProperties(properties));
      mavenConfiguration.getActiveProfiles().ifPresent(profiles -> defaultModelBuildingRequest.setActiveProfileIds(profiles));
      mavenConfiguration.getInactiveProfiles()
          .ifPresent(profiles -> defaultModelBuildingRequest.setInactiveProfileIds(profiles));
      defaultModelBuildingRequest.setTwoPhaseBuilding(false);
      defaultModelBuildingRequest.setProcessPlugins(false);
      defaultModelBuildingRequest.setModelCache(DefaultModelCache.newInstance(repositoryState.getSession()));
      defaultModelBuildingRequest
          .setModelResolver(repositoryState.createModelResolver(muleMavenResolutionContext.getRemoteRepositories()));
      defaultModelBuildingRequest.setSystemProperties(getProperties());
      defaultModelBuildingRequest.setValidationLevel(VALIDATION_LEVEL_MINIMAL);

      model = new DefaultModelBuilderFactory().newInstance().build(defaultModelBuildingRequest).getEffectiveModel();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
    return model;
  }

}
