/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import org.apache.maven.model.building.ModelCache;
import org.eclipse.aether.RepositoryCache;
import org.eclipse.aether.RepositorySystemSession;

/**
 * Implementation that delegates to {@link RepositorySystemSession#getCache()}.
 */
public class DefaultModelCache implements ModelCache {

  private final RepositorySystemSession session;

  private final RepositoryCache cache;

  public static ModelCache newInstance(RepositorySystemSession session) {
    if (session.getCache() == null) {
      return null;
    } else {
      return new DefaultModelCache(session);
    }
  }

  private DefaultModelCache(RepositorySystemSession session) {
    this.session = session;
    this.cache = session.getCache();
  }

  public Object get(String groupId, String artifactId, String version, String tag) {
    return cache.get(session, new DefaultModelCache.Key(groupId, artifactId, version, tag));
  }

  public void put(String groupId, String artifactId, String version, String tag, Object data) {
    cache.put(session, new DefaultModelCache.Key(groupId, artifactId, version, tag), data);
  }

  static class Key {

    private final String groupId;

    private final String artifactId;

    private final String version;

    private final String tag;

    private final int hash;

    public Key(String groupId, String artifactId, String version, String tag) {
      this.groupId = groupId;
      this.artifactId = artifactId;
      this.version = version;
      this.tag = tag;

      int h = 17;
      h = h * 31 + this.groupId.hashCode();
      h = h * 31 + this.artifactId.hashCode();
      h = h * 31 + this.version.hashCode();
      h = h * 31 + this.tag.hashCode();
      hash = h;
    }

    @Override
    public boolean equals(Object obj) {
      if (this == obj) {
        return true;
      }
      if (null == obj || !getClass().equals(obj.getClass())) {
        return false;
      }

      DefaultModelCache.Key that = (DefaultModelCache.Key) obj;
      return artifactId.equals(that.artifactId) && groupId.equals(that.groupId)
          && version.equals(that.version) && tag.equals(that.tag);
    }

    @Override
    public int hashCode() {
      return hash;
    }

  }
}
