/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.io.File.separator;
import static java.lang.String.format;
import static org.mule.maven.client.internal.util.FileUtils.getUrlsWithinJar;
import static org.mule.maven.client.internal.util.FileUtils.loadFileContentFrom;
import static org.mule.maven.client.internal.util.Preconditions.checkState;
import org.mule.maven.client.api.BundleDescriptorCreationException;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

public class MavenUtils {

  private static final String META_INF = "META-INF";
  private static final String MULE_ARTIFACT_FOLDER = META_INF + separator + "mule-artifact";
  private static final String MULE_PLUGIN_POM = "pom.xml";

  /**
   * Finds the URL of the pom file within the artifact file.
   *
   * @param artifactFile the artifact file to search for the pom file.
   * @return the URL to the pom file.
   */
  public static URL getPomUrlFromJar(File artifactFile) {
    try {
      List<URL> jarMavenUrls = getUrlsWithinJar(artifactFile, META_INF + "/" + "maven");
      Optional<URL> pomUrl = jarMavenUrls.stream().filter(url -> url.toString().endsWith("pom.xml")).findAny();
      if (!pomUrl.isPresent()) {
        throw new RuntimeException(format("The file 'pom.xml' was missing while looking within the artifact %s (it should be placed under [META-INF/maven/${groupId}/${artifactId}/pom.xml])",
                                          artifactFile.getName()));
      }
      return pomUrl.get();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * @param artifact a maven artifact that may be a directory following the mule deployable archive structure or a compressed
   *                 file.
   * @return the pom model for the artifact.
   */
  public static Model getPomModel(File artifact) {
    if (artifact.isDirectory()) {
      return getPomModelFolder(artifact);
    }
    return getPomModelFromJar(artifact);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param artifactFolder folder containing the exploded artifact file.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the folder does not contain a pom file or the file can' be loaded
   */
  private static Model getPomModelFolder(File artifactFolder) {
    final File mulePluginPom = lookupPomFromMavenLocation(artifactFolder);
    return getPomModelFromFile(mulePluginPom);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param artifactFile file containing the artifact content.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the artifact jar does not contain a pom file or the file can' be loaded
   */
  private static Model getPomModelFromJar(File artifactFile) {
    String pomFilePath = MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM;
    try {
      MavenXpp3Reader reader = new MavenXpp3Reader();
      return reader.read(new ByteArrayInputStream(loadFileContentFrom(getPomUrlFromJar(artifactFile)).get()));
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' for the artifact '%s'",
                                                         pomFilePath, artifactFile.getAbsolutePath()),
                                                  e);
    }
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param pomFile file containing the pom content.
   * @return the {@link Model} from the pom file
   * @throws BundleDescriptorCreationException if the file does not exist or the file can't be loaded
   */
  public static Model getPomModelFromFile(File pomFile) {
    MavenXpp3Reader reader = new MavenXpp3Reader();
    Model model;
    try (FileReader mulePluginPomFilerReader = new FileReader(pomFile)) {
      model = reader.read(mulePluginPomFilerReader);
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' in '%s'",
                                                         pomFile.getName(), pomFile.getParentFile().getAbsolutePath()),
                                                  e);
    }
    return model;
  }

  private static File lookupPomFromMavenLocation(File artifactFolder) {
    File mulePluginPom = null;
    File lookupFolder = new File(artifactFolder, "META-INF" + File.separator + "maven");
    while (lookupFolder != null && lookupFolder.exists()) {
      File possiblePomLocation = new File(lookupFolder, MULE_PLUGIN_POM);
      if (possiblePomLocation.exists()) {
        mulePluginPom = possiblePomLocation;
        break;
      }
      File[] directories = lookupFolder.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
      checkState(directories != null || directories.length == 0,
                 format("No directories under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      checkState(directories.length == 1,
                 format("More than one directory under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      lookupFolder = directories[0];
    }

    // final File mulePluginPom = new File(artifactFolder, MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM);
    if (mulePluginPom == null || !mulePluginPom.exists()) {
      throw new BundleDescriptorCreationException(format("The maven bundle loader requires the file pom.xml (error found while reading artifact '%s')",
                                                         artifactFolder.getName()));
    }
    return mulePluginPom;
  }

}
