/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.wagon;

import java.io.File;
import java.util.Properties;

import org.apache.maven.wagon.Wagon;
import org.apache.maven.wagon.providers.http.HttpWagon;
import org.apache.maven.wagon.shared.http.BasicAuthScope;
import org.apache.maven.wagon.shared.http.HttpConfiguration;
import org.apache.maven.wagon.shared.http.HttpMethodConfiguration;
import org.codehaus.plexus.component.configurator.ComponentConfigurationException;
import org.codehaus.plexus.component.configurator.converters.ConfigurationConverter;
import org.codehaus.plexus.component.configurator.converters.composite.PropertiesConverter;
import org.codehaus.plexus.component.configurator.expression.ExpressionEvaluationException;
import org.codehaus.plexus.component.configurator.expression.ExpressionEvaluator;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.xml.XmlPlexusConfiguration;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.eclipse.aether.transport.wagon.WagonConfigurator;

/**
 * An implementation of a {@link WagonConfigurator} which creates and supports those providers we need. Aether itself only
 * provides PlexusWagonConfigurator which uses Guice and we should keep this lightweight.
 */
public class SimplePlexusWagonConfigurator implements WagonConfigurator {

  private static ExpressionEvaluator NO_OP_EXPRESSION_EVALUATOR = new ExpressionEvaluator() {

    @Override
    public Object evaluate(String s) throws ExpressionEvaluationException {
      return s;
    }

    @Override
    public File alignToBaseDirectory(File file) {
      return file;
    }
  };

  public void configure(Wagon wagon, Object configuration) throws Exception {
    PlexusConfiguration config;
    if (configuration instanceof PlexusConfiguration) {
      config = (PlexusConfiguration) configuration;
    } else {
      if (!(configuration instanceof Xpp3Dom)) {
        if (configuration == null) {
          return;
        }

        throw new IllegalArgumentException("Unexpected configuration type: " + configuration.getClass().getName());
      }

      config = new XmlPlexusConfiguration((Xpp3Dom) configuration);
    }

    PropertiesConverter propertiesConverter = new PropertiesConverter();

    if (wagon instanceof HttpWagon) {
      HttpWagon httpWagon = (HttpWagon) wagon;
      for (PlexusConfiguration plexusConfiguration : config.getChildren()) {
        if ("httpHeaders".equals(plexusConfiguration.getName())) {
          httpWagon.setHttpHeaders(convert(plexusConfiguration, Properties.class, propertiesConverter));
        }
        if ("timeout".equals(plexusConfiguration.getName())) {
          httpWagon.setTimeout(Integer.valueOf(plexusConfiguration.getValue()));
        }
        if ("httpConfiguration".equals(plexusConfiguration.getName())) {
          httpConfiguration(httpWagon, plexusConfiguration, propertiesConverter);
        }
        if ("basicAuthScope".equals(plexusConfiguration.getName())) {
          BasicAuthScope basicAuthScope = new BasicAuthScope();
          PlexusConfiguration hostPlexusConfiguration = plexusConfiguration.getChild("host", false);
          if (hostPlexusConfiguration != null) {
            basicAuthScope.setHost(hostPlexusConfiguration.getValue());
          }
          PlexusConfiguration portPlexusConfiguration = plexusConfiguration.getChild("port", false);
          if (portPlexusConfiguration != null) {
            basicAuthScope.setPort(portPlexusConfiguration.getValue());
          }
          PlexusConfiguration realmPlexusConfiguration = plexusConfiguration.getChild("realm", false);
          if (realmPlexusConfiguration != null) {
            basicAuthScope.setRealm(realmPlexusConfiguration.getValue());
          }
          httpWagon.setBasicAuthScope(basicAuthScope);
        }
      }
    }
  }

  private void httpConfiguration(HttpWagon httpWagon, PlexusConfiguration plexusConfiguration,
                                 PropertiesConverter propertiesConverter)
      throws ComponentConfigurationException {
    HttpConfiguration httpConfiguration = new HttpConfiguration();
    for (PlexusConfiguration methodPlexusConfiguration : plexusConfiguration.getChildren()) {
      switch (methodPlexusConfiguration.getName()) {
        case "all":
          httpConfiguration.setAll(httpMethodConfiguration(plexusConfiguration.getChild("all"), propertiesConverter));
          break;
        case "get":
          httpConfiguration.setGet(httpMethodConfiguration(plexusConfiguration.getChild("get"), propertiesConverter));
          break;
        case "head":
          httpConfiguration.setHead(httpMethodConfiguration(plexusConfiguration.getChild("head"), propertiesConverter));
          break;
        case "put":
          httpConfiguration.setPut(httpMethodConfiguration(plexusConfiguration.getChild("put"), propertiesConverter));
          break;
        default:
          break;
      }
    }
    httpWagon.setHttpConfiguration(httpConfiguration);
  }

  private HttpMethodConfiguration httpMethodConfiguration(PlexusConfiguration plexusConfiguration,
                                                          PropertiesConverter propertiesConverter)
      throws ComponentConfigurationException {
    HttpMethodConfiguration config = new HttpMethodConfiguration();

    config.setUsePreemptive(Boolean.valueOf(plexusConfiguration.getChild("usePreemptive").getValue()));

    config.setUseDefaultHeaders(Boolean.valueOf(plexusConfiguration.getChild("useDefaultHeaders").getValue()));

    PlexusConfiguration connectionTimeout = plexusConfiguration.getChild("connectionTimeout", false);
    if (connectionTimeout != null) {
      config.setConnectionTimeout(Integer.valueOf(connectionTimeout.getValue()));
    }

    PlexusConfiguration readTimeout = plexusConfiguration.getChild("readTimeout", false);
    if (readTimeout != null) {
      config.setReadTimeout(Integer.valueOf(readTimeout.getValue()));
    }

    PlexusConfiguration params = plexusConfiguration.getChild("params", false);
    if (params != null) {
      config.setParams(convert(params, Properties.class, propertiesConverter));
    }

    PlexusConfiguration headers = plexusConfiguration.getChild("headers", false);
    if (headers != null) {
      config.setHeaders(convert(headers, Properties.class, propertiesConverter));
    }

    return config;
  }

  private static <T> T convert(PlexusConfiguration plexusConfiguration, Class<T> clazz,
                               ConfigurationConverter configurationConverter)
      throws ComponentConfigurationException {
    return (T) configurationConverter.fromConfiguration(null, plexusConfiguration, clazz, null,
                                                        SimplePlexusWagonConfigurator.class.getClassLoader(),
                                                        NO_OP_EXPRESSION_EVALUATOR);
  }

}
