/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static java.lang.Boolean.TRUE;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.newSetFromMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.eclipse.aether.util.artifact.ArtifactIdUtils.toId;
import static org.eclipse.aether.util.artifact.JavaScopes.COMPILE;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.mule.maven.client.internal.AetherMavenClient.MULE_PLUGIN_CLASSIFIER;
import static org.mule.maven.client.internal.ApiDependencyGraphTransformer.isApi;
import static org.mule.maven.client.internal.MulePluginDependencyGraphTransformer.isPlugin;

import org.mule.maven.client.internal.ApiDependencyGraphTransformer;
import org.mule.maven.client.internal.OneInstancePerNodeGraphTransformer;

import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.graph.DefaultDependencyNode;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.util.graph.visitor.CloningDependencyVisitor;
import org.hamcrest.Description;
import org.hamcrest.TypeSafeMatcher;
import org.junit.Test;

public class OneInstancePerNodeGraphTransformerTestCase {

  private static final DependencyGraphTransformer TRANSFORMER = new OneInstancePerNodeGraphTransformer();

  private static final String JAR = "jar";
  private static final String RAML_CLASSIFIER = "raml";
  private static final String DEFAULT_GROUP_ID = "org.mule.test";
  private static final String DEFAULT_VERSION = "1.0.0";

  private static List<DependencyNode> collectContextSubGraph(DependencyNode root) {
    Set<DependencyNode> cache = newSetFromMap(new IdentityHashMap<>());
    List<DependencyNode> nodes = collectContextChildNodes(root, cache);
    nodes.add(0, root);
    return nodes;
  }

  private static List<DependencyNode> collectContextChildNodes(DependencyNode root, Set<DependencyNode> cache) {
    return collectChildNodes(root, cache, c -> !isPlugin(c));
  }

  private static List<DependencyNode> collectChildNodes(DependencyNode root, Set<DependencyNode> cache,
                                                        Predicate<DependencyNode> filter) {
    if (cache.contains(root)) {
      return emptyList();
    }
    cache.add(root);
    List<DependencyNode> nodes = new LinkedList<>();
    root.getChildren().stream().filter(filter).forEach(fc -> {
      nodes.add(fc);
      nodes.addAll(collectChildNodes(fc, cache, filter));
    });
    return nodes;
  }

  private static void compareGraphs(DependencyNode root1, DependencyNode root2, Set<DependencyNode> cache) {
    if (cache.contains(root1) && cache.contains(root2)) {
      return;
    }
    cache.add(root1);
    cache.add(root2);

    assertThat(root1, NodeMatcher.sameNode(root2));
    assertThat(root1.getChildren().size(), equalTo(root2.getChildren().size()));
    for (int i = 0; i < root1.getChildren().size(); i++) {
      compareGraphs(root1.getChildren().get(i),
                    root2.getChildren().get(i),
                    cache);
    }
  }

  @Test
  public void simpleGraphRemainsTheSame() throws RepositoryException {
    DependencyNode root =
        dependency("a",
                   dependency("b1",
                              dependency("c11"),
                              dependency("c12")),
                   dependency("b2",
                              dependency("c21"),
                              dependency("c22")));
    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(1));
  }

  @Test
  public void simpleGraphKeepsSameInstances() throws RepositoryException {
    DependencyNode libA = dependency("a",
                                     dependency("a1"),
                                     dependency("a2"));
    DependencyNode root = dependency("root",
                                     dependency("b", libA),
                                     dependency("c", libA));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(1));

    List<List<DependencyNode>> subgraphs = splitGraph.getContextSubgraphsFor("b", "c");
    assertThat(subgraphs.size(), equalTo(2));
    validateSameInstancesBetweenContexts(subgraphs);

  }

  @Test
  public void simpleRecursiveGraph() throws RepositoryException {
    DependencyNode libC = dependency("c");
    DependencyNode libB = dependency("b", libC);
    DependencyNode libA = dependency("a", libB);
    libC.setChildren(singletonList(libA));

    DependencyNode root = dependency("root",
                                     dependency("first-level", libA));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(1));

    List<List<DependencyNode>> subgraphs = splitGraph.getContextSubgraphsFor("a", "b", "c");
    assertThat(subgraphs.size(), equalTo(3));
    validateSameInstancesBetweenContexts(subgraphs);
  }

  @Test
  public void multiplePluginsDifferentDependencies() throws RepositoryException {
    DependencyNode root = dependency("root",
                                     plugin("plugin-a", dependency("transitive-a")),
                                     plugin("plugin-b", dependency("transitive-b")),
                                     plugin("plugin-c", dependency("transitive-c")));
    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(4));
  }

  @Test
  public void differentInstancesForEachContext() throws RepositoryException {
    DependencyNode dep = dependency("dependency",
                                    dependency("transitive1"),
                                    dependency("transitive2"));

    DependencyNode root = dependency("root",
                                     dependency("a", dep),
                                     dependency("b", dep),
                                     plugin("plugin1", dep),
                                     plugin("plugin2", dep));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(3));
    List<List<DependencyNode>> subgraphs = splitGraph.getContextSubgraphsFor("a", "b");
    assertThat(subgraphs.size(), equalTo(2));
    validateSameInstancesBetweenContexts(subgraphs);
  }

  @Test
  public void multipleOccurencesOfSamePluginsAreAllDifferentInstances() throws RepositoryException {
    DependencyNode dep = dependency("dependency",
                                    dependency("transitive1"),
                                    dependency("transitive2"));

    DependencyNode pluginv1 = plugin("plugin", "1.0.0", dep);
    DependencyNode pluginv2 = plugin("plugin", "2.0.0", dep);
    DependencyNode root = dependency("root",
                                     pluginv1,
                                     pluginv2,
                                     pluginv1,
                                     pluginv2);

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(5));
  }

  @Test
  public void cyclesBetweenPluginsAndNonPlugins() throws RepositoryException {
    DependencyNode libC = dependency("c");
    DependencyNode libB = dependency("b", libC);
    DependencyNode libA = dependency("a", libB);
    libC.setChildren(singletonList(libA));

    DependencyNode root = dependency("root",
                                     dependency("non-plugin", libA),
                                     plugin("plugin", libA));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(2));
  }

  @Test
  public void multiplePluginLevels() throws RepositoryException {
    DependencyNode libA = dependency("a",
                                     dependency("transitive1"),
                                     dependency("transitive2"));

    DependencyNode root = dependency("root",
                                     dependency("non-plugin",
                                                libA,
                                                dependency("non-plugin2",
                                                           plugin("plugin3", libA))),
                                     plugin("plugin",
                                            libA,
                                            plugin("plugin2",
                                                   libA)));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(4));
    List<List<DependencyNode>> subgraphs = splitGraph.getContextSubgraphsFor("non-plugin", "non-plugin2");
    assertThat(subgraphs.size(), equalTo(2));
    validateSameInstancesBetweenContexts(subgraphs);
  }

  @Test
  public void pluginsCyclesToNonPlugin() throws RepositoryException {
    DependencyNode nonPlugin = dependency("a",
                                          dependency("b",
                                                     dependency("c")));
    DependencyNode root = dependency("root",
                                     nonPlugin,
                                     dependency("otherNonPlugin",
                                                plugin("plugin1", nonPlugin, plugin(
                                                                                    "plugin2", nonPlugin))));

    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(3));
  }

  @Test
  public void multipleApiDependencies() throws RepositoryException {
    DependencyNode simpleDeps = dependency("a",
                                           dependency("b1"),
                                           dependency("b2"));
    DependencyNode fragment = api("fragment");
    DependencyNode root = dependency("root",
                                     simpleDeps,
                                     api("api1", simpleDeps, fragment),
                                     api("api2", simpleDeps, fragment));
    SplitGraph splitGraph = transformAndCheck(root);
    assertThat(splitGraph.getContextCount(), is(1));

    List<List<DependencyNode>> subgraphs = splitGraph.getContextSubgraphsFor("api1", "api2", "root");
    assertThat(subgraphs, hasSize(3));
    validateSameInstancesBetweenContexts(subgraphs, d -> !isApi(d));
  }

  private SplitGraph transformAndCheck(DependencyNode root) throws RepositoryException {
    // Clone the whole graph so that we can keep the original data
    CloningDependencyVisitor cloner = new CloningDependencyVisitor();
    root.accept(cloner);

    DependencyNode clonedRoot = cloner.getRootNode();

    // Transform the graph and flatten it's result.
    DependencyNode transformedRoot = TRANSFORMER.transformGraph(clonedRoot, null);

    // Both transformed and original graphs should be the same, the only thing that should have changed is that some nodes
    // should've
    // been re instantiated, but their information should be the same.
    compareGraphs(root, transformedRoot, newSetFromMap(new IdentityHashMap<>()));

    // Split the graph so that we are able to collect nodes by context.
    SplitGraph splitGraph = new SplitGraph(transformedRoot);

    // Check that there is no repeated instance between any collected subgraph.
    validateDifferentInstancesForEachContext(splitGraph.getAllContextSubgraphs(), d -> true); // Mp filter, check all dependencies
    validateDifferentInstancesForEachContext(splitGraph.getApiSubgraphs(), ApiDependencyGraphTransformer::isApi); // only check
                                                                                                                  // api
                                                                                                                  // dependencies

    return splitGraph;
  }

  private void validateDifferentInstancesForEachContext(List<List<DependencyNode>> contexts,
                                                        Predicate<DependencyNode> instanceFilter) {
    // Compare each pair of contexts and check that if they have the same dependency node, then it's a different instance
    for (List<DependencyNode> c1 : contexts) {
      for (List<DependencyNode> c2 : contexts) {
        if (c1 == c2) {
          continue;
        }
        Set<DependencyNode> intersection = newSetFromMap(new IdentityHashMap<>());
        intersection.addAll(c1.stream().filter(instanceFilter).collect(toList()));
        intersection.retainAll(c2.stream().filter(instanceFilter).collect(toList()));
        assertThat(intersection, hasSize(0));
      }
    }
  }

  private void validateSameInstancesBetweenContexts(List<List<DependencyNode>> contexts) {
    validateSameInstancesBetweenContexts(contexts, d -> true);
  }

  private void validateSameInstancesBetweenContexts(List<List<DependencyNode>> contexts,
                                                    Predicate<DependencyNode> instanceFilter) {
    // Check that if the same dependency is in any of the received contexts, then they are all the same instance.
    // Filter out any instance that we don't want to check, like api dependencies for api contexts
    Map<String, List<DependencyNode>> nodes = new HashMap<>();
    contexts.forEach(
                     c -> c.stream().filter(instanceFilter).forEach(
                                                                    n -> {
                                                                      String id = toId(n.getArtifact());
                                                                      nodes.computeIfAbsent(id, k -> new LinkedList<>());
                                                                      nodes.get(id).add(n);
                                                                    }));
    nodes.values().forEach(
                           nodeList -> {
                             Set<DependencyNode> set = newSetFromMap(new IdentityHashMap<>());
                             set.addAll(nodeList);
                             assertThat(set, hasSize(1));
                           });
  }

  private DependencyNode plugin(String artifactId, String version, String groupId, DependencyNode... children) {
    return buildNode(new DefaultArtifact(groupId, artifactId, MULE_PLUGIN_CLASSIFIER, JAR, version), children);
  }

  private DependencyNode plugin(String artifactId, String version, DependencyNode... children) {
    return plugin(artifactId, version, DEFAULT_GROUP_ID, children);
  }

  private DependencyNode plugin(String artifactId, DependencyNode... children) {
    return plugin(artifactId, DEFAULT_VERSION, children);
  }

  private DependencyNode api(String artifactId, String version, String groupId, DependencyNode... children) {
    return buildNode(new DefaultArtifact(groupId, artifactId, RAML_CLASSIFIER, JAR, version), children);
  }

  private DependencyNode api(String artifactId, String version, DependencyNode... children) {
    return api(artifactId, version, DEFAULT_GROUP_ID, children);
  }

  private DependencyNode api(String artifactId, DependencyNode... children) {
    return api(artifactId, DEFAULT_VERSION, children);
  }

  private DependencyNode dependency(String artifactId, String version, String groupId, DependencyNode... children) {
    return buildNode(new DefaultArtifact(groupId, artifactId, JAR, version), children);
  }

  private DependencyNode dependency(String artifactId, String version, DependencyNode... children) {
    return dependency(artifactId, version, DEFAULT_GROUP_ID, children);
  }

  private DependencyNode dependency(String artifactId, DependencyNode... children) {
    return dependency(artifactId, DEFAULT_VERSION, children);
  }

  private DependencyNode buildNode(Artifact artifact, DependencyNode... children) {
    Dependency dependency = new Dependency(artifact, COMPILE);
    DependencyNode node = new DefaultDependencyNode(dependency);
    node.setChildren(asList(children));
    return node;
  }

  /**
   * {@link org.hamcrest.Matcher} to compare {@link DependencyNode}s.
   *
   * WARNING: Only the coordinates of the node are checked, this matcher does not evaluate node data or children.
   */
  private static class NodeMatcher extends TypeSafeMatcher<DependencyNode> {

    private DependencyNode expected;

    public static NodeMatcher sameNode(DependencyNode expected) {
      return new NodeMatcher(expected);
    }

    private NodeMatcher(DependencyNode expected) {
      this.expected = expected;
    }

    @Override
    protected boolean matchesSafely(DependencyNode item) {
      Artifact itemArtifact = item.getArtifact();
      Artifact expectedArtifact = expected.getArtifact();

      return itemArtifact.getArtifactId().equals(expectedArtifact.getArtifactId()) &&
          itemArtifact.getGroupId().equals(expectedArtifact.getGroupId()) &&
          itemArtifact.getBaseVersion().equals(expectedArtifact.getBaseVersion());
    }

    @Override
    public void describeTo(Description description) {
      description.appendText("are not same node");
    }
  }

  /**
   * POJO to store nodes from a graph, splitting them according to their classifier.
   */
  private static class SplitGraph {

    private static final String VISITED = "visited";

    // All mule-plugin nodes
    private List<DependencyNode> pluginRootNodes = new LinkedList<>();

    // All nodes without mule-plugin as parent
    private List<DependencyNode> nonPluginRootNodes = new LinkedList<>();

    // All raml nodes without mule-plugin as parent
    private List<DependencyNode> apiRootNodes = new LinkedList<>();

    private SplitGraph(DependencyNode root) {
      collect(root, false, false);
    }

    /**
     * We need to validate that if the same dependency appears in a plugin subgraph, then it should be a different instance for
     * each plugin and for the main subgraph too. (The main subgraph would be the graph formed by removing all mule-plugin nodes
     * and it's children.)
     *
     * Also, we should collect api nodes because we need to know that all their instances are different. Api children should be
     * considered as part of the 'main subgraph' as they will need to be part of it's conflict resolution and they will need to
     * keep cycles.
     */
    private void collect(DependencyNode node, boolean isPluginChild, boolean isApiChild) {
      if (node.getData().containsKey(VISITED)) {
        return;
      }
      node.setData(VISITED, TRUE);
      if (isPlugin(node)) {
        pluginRootNodes.add(node);
        isPluginChild = true;
      } else if (isApi(node) && !isApiChild && !isPluginChild) {
        apiRootNodes.add(node);
        nonPluginRootNodes.add(node);
        isApiChild = true;
      } else {
        if (!isPluginChild) {
          nonPluginRootNodes.add(node);
        }
      }
      final boolean pluginChild = isPluginChild;
      final boolean apiChild = isApiChild;
      node.getChildren().forEach(c -> collect(c, pluginChild, apiChild));
    }

    private List<List<DependencyNode>> getPluginSubgraphs() {
      List<List<DependencyNode>> pluginSubgraph = new LinkedList<>();
      pluginRootNodes.forEach(t -> pluginSubgraph.add(collectContextSubGraph(t)));
      return pluginSubgraph;
    }

    private List<DependencyNode> getMainSubgraph() {
      return nonPluginRootNodes;
    }

    private List<List<DependencyNode>> getApiSubgraphs() {
      List<List<DependencyNode>> apiSubgraphs = new LinkedList<>();
      apiRootNodes.forEach(t -> apiSubgraphs.add(collectContextSubGraph(t)));
      return apiSubgraphs;
    }

    private List<List<DependencyNode>> getAllContextSubgraphs() {
      List<List<DependencyNode>> subgraphs = getPluginSubgraphs();
      subgraphs.add(nonPluginRootNodes);
      return subgraphs;
    }

    private List<List<DependencyNode>> getContextSubgraphsFor(String... rootNodes) {
      List<List<DependencyNode>> subGraphs = new LinkedList<>();
      for (String node : rootNodes) {
        pluginRootNodes.stream().filter(pn -> pn.getArtifact().getArtifactId().equals(node))
            .forEach(n -> subGraphs.add(collectContextSubGraph(n)));
        nonPluginRootNodes.stream().filter(pn -> pn.getArtifact().getArtifactId().equals(node))
            .forEach(n -> subGraphs.add(collectContextSubGraph(n)));
      }
      return subGraphs;
    }

    private int getContextCount() {
      return getAllContextSubgraphs().size();
    }
  }

}
