/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal.util;

import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_FAIL;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_ALWAYS;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_INTERVAL;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.mule.maven.client.api.model.RepositoryPolicy.newRepositoryPolicyBuilder;
import static org.mule.maven.client.internal.AetherRepositoryStateFactory.newAetherRepositoryState;

import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.xml.XmlPlexusConfiguration;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.AetherRepositoryState;
import org.mule.maven.client.internal.AetherResolutionContext;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Optional;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class AetherRepositoryStateTestCase {

  private static final String DOWNLOADS_PLANETMIRROR_URL = "http://downloads.planetmirror.com/pub/maven2";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Test
  public void globalChecksumPolicyTest() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id("serverId")
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .releasePolicy(newRepositoryPolicyBuilder()
                .enabled(true)
                .checksumPolicy(CHECKSUM_POLICY_FAIL)
                .build())
            .build())
        .globalChecksumPolicy(CHECKSUM_POLICY_WARN)
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    assertThat(context.getRemoteRepositories().get(0).getPolicy(false).getChecksumPolicy(), equalTo(CHECKSUM_POLICY_FAIL));
    AetherRepositoryState aetherRepositoryState =
        newAetherRepositoryState(temporaryFolder.getRoot(), context, mavenConfiguration, Optional.empty(), Optional.empty());
    assertThat(aetherRepositoryState.getSession().getChecksumPolicy(), equalTo(CHECKSUM_POLICY_WARN));
  }

  @Test
  public void forceUpdatePolicyNeverTest() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id("serverId")
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .releasePolicy(newRepositoryPolicyBuilder()
                .enabled(true)
                .updatePolicy(UPDATE_POLICY_INTERVAL + ":30")
                .build())
            .build())
        .forcePolicyUpdateNever(true)
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    assertThat(context.getRemoteRepositories().get(0).getPolicy(false).getUpdatePolicy(),
               equalTo(UPDATE_POLICY_INTERVAL + ":30"));
    AetherRepositoryState aetherRepositoryState =
        newAetherRepositoryState(temporaryFolder.getRoot(), context, mavenConfiguration, Optional.empty(), Optional.empty());
    assertThat(aetherRepositoryState.getSession().getUpdatePolicy(), equalTo(UPDATE_POLICY_NEVER));
  }

  @Test
  public void forceUpdatePolicyAlwaysTest() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id("serverId")
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .releasePolicy(newRepositoryPolicyBuilder()
                .enabled(true)
                .updatePolicy(UPDATE_POLICY_NEVER)
                .build())
            .build())
        .forcePolicyUpdateAlways(true)
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    assertThat(context.getRemoteRepositories().get(0).getPolicy(false).getUpdatePolicy(), equalTo(UPDATE_POLICY_NEVER));
    AetherRepositoryState aetherRepositoryState =
        newAetherRepositoryState(temporaryFolder.getRoot(), context, mavenConfiguration, Optional.empty(), Optional.empty());
    assertThat(aetherRepositoryState.getSession().getUpdatePolicy(), equalTo(UPDATE_POLICY_ALWAYS));
  }

  @Test
  public void serverXmlPlexusConfigurationTest() throws IOException {
    File localRepository = temporaryFolder.newFolder();
    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-server-wagon-configuration.xml");
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .userSettingsLocation(toFile(settingsUrl))
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id("serverId")
            .url(new URL(DOWNLOADS_PLANETMIRROR_URL))
            .build())
        .forcePolicyUpdateAlways(true)
        .build();

    AetherResolutionContext context = new AetherResolutionContext(mavenConfiguration);
    AetherRepositoryState aetherRepositoryState =
        newAetherRepositoryState(temporaryFolder.getRoot(), context, mavenConfiguration, Optional.empty(), Optional.empty());
    String serverConfigKey = "aether.connector.wagon.config.global-wagon-standards";
    assertThat(aetherRepositoryState.getSession().getConfigProperties().containsKey(serverConfigKey), is(true));
    assertThat(aetherRepositoryState.getSession().getConfigProperties().get(serverConfigKey),
               instanceOf(PlexusConfiguration.class));
    XmlPlexusConfiguration child = (XmlPlexusConfiguration) ((XmlPlexusConfiguration) aetherRepositoryState.getSession()
        .getConfigProperties().get(serverConfigKey))
            .getChild("httpHeaders").getChild("property");
    assertThat(child.getChild("name").getValue(), is("Authorization"));
    assertThat(child.getChild("value").getValue(), is("Basic {base64 encoded string}"));
  }

}
