/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.util.Objects.requireNonNull;

import java.util.List;

import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.impl.RemoteRepositoryManager;
import org.eclipse.aether.repository.RemoteRepository;

/**
 * Resolver the list of {@link RemoteRepository} to be used by {@link org.eclipse.aether.RepositorySystem} when
 * resolving dependencies taking into account Maven's logic to order repositories.
 */
public class RemoteRepositoryModelResolver {

  private RemoteRepositoryManager remoteRepositoryManager;
  private RepositorySystemSession repositorySystemSession;

  public RemoteRepositoryModelResolver(RemoteRepositoryManager remoteRepositoryManager,
                                       RepositorySystemSession repositorySystemSession) {
    requireNonNull(remoteRepositoryManager);
    requireNonNull(repositorySystemSession);

    this.remoteRepositoryManager = remoteRepositoryManager;
    this.repositorySystemSession = repositorySystemSession;
  }

  /**
   * Uses Maven's logic to sort the remote repositories in the correct order and optionally applies mirror, proxy and
   * authentication settings from the session.
   *
   * @param settingsRepositories original list of {@link RemoteRepository repositories} from settings.xml or {@link org.mule.maven.client.api.model.MavenConfiguration}, these
   *                             repositories should already have applied mirror, proxy and authentication.
   * @param modelRepositories    list of {@link RemoteRepository repositories} to be added if there is not already one in dominant list for the same ID.
   * @return {@link List} of {@link RemoteRepository} in the same order as Maven does when resolving dependencies.
   */
  public List<RemoteRepository> resolveRepositories(List<RemoteRepository> settingsRepositories,
                                                    List<RemoteRepository> modelRepositories) {
    // modelRepositories will come as raw (as they come from model or either a artifactDescriptorResult), so authenticator, mirror and proxy have to be set
    return remoteRepositoryManager.aggregateRepositories(repositorySystemSession, settingsRepositories,
                                                         modelRepositories,
                                                         true);
  }

}
