/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import org.eclipse.aether.RepositoryException;
import org.eclipse.aether.collection.DependencyGraphTransformationContext;
import org.eclipse.aether.collection.DependencyGraphTransformer;
import org.eclipse.aether.graph.DependencyNode;

import java.util.IdentityHashMap;
import java.util.Map;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.mule.maven.client.internal.AetherMavenClient.MULE_PLUGIN_CLASSIFIER;

/**
 * Transformer that will go over the graph and look for {@code mule-plugin} classified dependencies. Once found it will go over
 * its children and only leave the {@code mule-plugin} dependencies and remove any other one.
 * <p>
 * This way nested {@link DependencyGraphTransformer transformers} will apply Maven's logic and transitive dependencies from
 * {@code mule-plugin}s will not interfere.
 *
 * @since 1.0
 */
public class MulePluginDependencyGraphTransformer implements DependencyGraphTransformer {

  @Override
  public DependencyNode transformGraph(DependencyNode node, DependencyGraphTransformationContext context)
      throws RepositoryException {
    Map<DependencyNode, Object> visitedNodes = new IdentityHashMap<>(512);
    visitGraph(node, visitedNodes, true);
    return node;
  }

  private void visitGraph(DependencyNode node, Map<DependencyNode, Object> visitedNodes, boolean rootNode)
      throws RepositoryException {
    if (!setVisited(node, visitedNodes)) {
      return;
    }
    if (isPlugin(node) && !rootNode) {
      while (!hasOnlyPluginDependencies(node)) {
        removeNormalDependencies(node);
      }
    }

    for (DependencyNode child : node.getChildren()) {
      visitGraph(child, visitedNodes, false);
    }
  }

  /**
   * Marks the specified node as being visited and determines whether the node has been visited before.
   *
   * @param node The node being visited, must not be {@code null}.
   * @return {@code true} if the node has not been visited before, {@code false} if the node was already visited.
   */
  private boolean setVisited(DependencyNode node, Map<DependencyNode, Object> visitedNodes) {
    return visitedNodes.put(node, Boolean.TRUE) == null;
  }

  /**
   * @return {@code true} if the node has {@code mule-plugin} as children.
   */
  private boolean hasOnlyPluginDependencies(DependencyNode node) {
    return node.getChildren().isEmpty() || node.getChildren().stream().allMatch(this::isPlugin);
  }

  /**
   * From the node given it will remove any child node that is not a {@code mule-plugin}. It goes over the child's of the children
   * nodes.
   *
   * @param node the initial node to start removing nodes that are not {@code mule-plugin} as children.
   */
  private void removeNormalDependencies(DependencyNode node) {
    node.setChildren(node.getChildren()
        .stream()
        .flatMap(childNode -> {
          if (isPlugin(childNode)) {
            return asList(childNode).stream();
          } else {
            //TODO MMP-442: Add test cases and review solution
            return childNode.getChildren().stream().filter(n -> haveDifferentGroupAndArtifactIds(childNode, n));
          }
        })
        .collect(toList()));
  }

  /**
   * @return {@code true} if a and b are not the same dependency
   */
  private boolean haveDifferentGroupAndArtifactIds(DependencyNode a, DependencyNode b) {
    return !a.getArtifact().getGroupId().equals(b.getArtifact().getGroupId())
        || !a.getArtifact().getArtifactId().equals(b.getArtifact().getArtifactId());
  }

  /**
   * @return {@code true} if node is a {@code mule-plugin}.
   */
  private boolean isPlugin(DependencyNode node) {
    return node != null && node.getArtifact() != null && MULE_PLUGIN_CLASSIFIER.equals(node.getArtifact().getClassifier());
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    return null != obj && this.getClass().equals(obj.getClass());
  }

  @Override
  public int hashCode() {
    return this.getClass().hashCode();
  }

}
