/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import static java.io.File.separator;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.maven.client.internal.MavenEnvironmentVariables.MAVEN_CMD_LINE_ARGS;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.maven.client.api.SettingsSupplierFactory;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;

/**
 * Provides a set of suppliers to be used to resolve maven settings files.
 *
 * @since 1.0
 */
public class DefaultSettingsSupplierFactory implements SettingsSupplierFactory {

  private static final Logger LOGGER = getLogger(DefaultSettingsSupplierFactory.class);

  private static final String SETTINGS_XML = "settings.xml";
  private static final String SETTINGS_SECURITY_XML = "settings-security.xml";
  private static final String MAVEN_GLOBAL_SETTINGS_PATH = "conf" + separator + SETTINGS_XML;

  public static final String MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY = "settings.security";

  private MavenEnvironmentVariables mavenEnvironmentVariables;

  public DefaultSettingsSupplierFactory(MavenEnvironmentVariables mavenEnvironmentVariables) {
    requireNonNull(mavenEnvironmentVariables, "mavenEnvironmentVariables cannot be null");

    this.mavenEnvironmentVariables = mavenEnvironmentVariables;
  }

  @Override
  public Optional<File> environmentGlobalSettingsSupplier() {
    MavenArguments mavenArguments = parseMavenArguments();

    if (mavenArguments.getGlobalSettings().isPresent()) {
      final File file = mavenArguments.getGlobalSettings().get();
      if (file.exists()) {
        logGlobalSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(mavenArguments.getGlobalSettings().get());
      }
    }

    File globalSettingsFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY);
    if (globalSettingsFile != null) {
      logGlobalSettings(globalSettingsFile, GLOBAL_SETTINGS_SYSTEM_PROPERTY);
      return of(globalSettingsFile);
    }

    String mavenHome = mavenEnvironmentVariables.getM2HomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeEnv();
    mavenHome = mavenHome != null ? mavenHome : mavenEnvironmentVariables.getMavenHomeProperty();
    if (mavenHome != null) {
      File globalSettings = new File(mavenHome, MAVEN_GLOBAL_SETTINGS_PATH);
      if (globalSettings.exists()) {
        logGlobalSettings(globalSettings, mavenHome);
        return of(globalSettings);
      }
    }
    return empty();
  }

  @Override
  public Optional<File> environmentUserSettingsSupplier() {
    MavenArguments mavenArguments = parseMavenArguments();

    if (mavenArguments.getSettings().isPresent()) {
      final File file = mavenArguments.getSettings().get();
      if (file.exists()) {
        logUserSettings(file, MAVEN_CMD_LINE_ARGS);
        return of(file);
      }
    }

    File userSettingsFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY);
    if (userSettingsFile != null) {
      logUserSettings(userSettingsFile, USER_SETTINGS_SYSTEM_PROPERTY);
      return of(userSettingsFile);
    }

    File userM2Folder = new File(mavenEnvironmentVariables.getUserHome(), ".m2");
    if (userM2Folder.exists()) {
      final File file = new File(userM2Folder, SETTINGS_XML);
      if (file.exists()) {
        logUserSettings(file, userM2Folder.getAbsolutePath());
        return of(file);
      }
    }
    return empty();
  }

  @Override
  public Optional<File> environmentSettingsSecuritySupplier() {
    File settingsSecurityFile = mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY);
    if (settingsSecurityFile != null) {
      logSettingsSecurity(settingsSecurityFile, MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY);
      return of(settingsSecurityFile);
    }

    File userM2Folder = new File(mavenEnvironmentVariables.getUserHome(), ".m2");
    if (userM2Folder.exists()) {
      final File file = new File(userM2Folder, SETTINGS_SECURITY_XML);
      if (file.exists()) {
        logUserSettings(file, userM2Folder.getAbsolutePath());
        return of(file);
      }
    }
    return empty();
  }

  private void logSettingsSecurity(File settingsSecurity, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven settings security '{}' from '{}'", settingsSecurity.getAbsolutePath(), location);
    }
  }

  private void logUserSettings(File userSettings, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven user settings '{}' from '{}'", userSettings.getAbsolutePath(), location);
    }
  }

  private void logGlobalSettings(File globalSettings, String location) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Resolved Maven global settings '{}' from '{}'", globalSettings.getAbsolutePath(), location);
    }
  }

  private String[] getMavenCmdLineArgs() {
    String mavenCmdLineArgs = mavenEnvironmentVariables.getMavenCmdLineArgsEnv();
    if (mavenCmdLineArgs != null) {
      return mavenCmdLineArgs.trim().split(" ");
    }
    return new String[0];
  }

  private MavenArguments parseMavenArguments() {
    MavenArguments mavenArguments = new MavenArguments();
    JCommander.newBuilder()
        .addObject(mavenArguments)
        .acceptUnknownOptions(true)
        .allowParameterOverwriting(true)
        .build()
        .parse(getMavenCmdLineArgs());
    return mavenArguments;
  }

  /**
   * POJO to be created by JCommander when parsing the command line arguments from Maven.
   * It supports as Maven command line duplicate entries for settings and global settings which in that case uses the first
   * one.
   */
  private class MavenArguments {

    @Parameter(names = {"-s", "--settings"}, description = "User settings")
    private List<File> settings = new ArrayList<>();

    @Parameter(names = {"-gs", "--global-settings"}, description = "Global settings")
    private List<File> globalSettings = new ArrayList<>();

    public Optional<File> getSettings() {
      return ofNullable(settings.size() >= 1 ? settings.get(0) : null);
    }

    public Optional<File> getGlobalSettings() {
      return ofNullable(globalSettings.size() >= 1 ? globalSettings.get(0) : null);
    }
  }

}
