/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.internal;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.graph.DependencyNode;

import java.util.HashMap;
import java.util.Map;

import static org.eclipse.aether.util.StringUtils.isEmpty;

/**
 * Utility class that converts a dependency node into a BundleDependency by also converting the transitive dependencies and
 * caching the results for future usages.
 */
public class BundleDependencyHelper {

  private Map<BundleDescriptor, BundleDependency> dependenciesMap = new HashMap<>();

  public BundleDependency getBundleDependency(DependencyNode node) {
    Artifact artifact = node.getArtifact();
    BundleDescriptor bundleDescriptor = artifactToBundleDescriptor(artifact).build();
    BundleDependency cachedValue = dependenciesMap.get(bundleDescriptor);
    if (cachedValue != null) {
      return cachedValue;
    }
    String scope = node.getDependency().getScope();

    BundleScope bundleScope = StringUtils.isEmpty(scope) ? BundleScope.COMPILE : BundleScope.valueOf(scope.toUpperCase());
    BundleDependency.Builder bundleDependencyBuilder = artifactToBundleDependency(node.getArtifact(), bundleScope.name());
    node.getChildren().stream()
        .forEach(childNode -> bundleDependencyBuilder.addTransitiveDependency(getBundleDependency(childNode)));
    return bundleDependencyBuilder.build();
  }

  static BundleDependency.Builder artifactToBundleDependency(Artifact artifact, String scope) {
    final BundleDescriptor.Builder bundleDescriptorBuilder = artifactToBundleDescriptor(artifact);

    BundleScope bundleScope = StringUtils.isEmpty(scope) ? BundleScope.COMPILE : BundleScope.valueOf(scope.toUpperCase());
    BundleDependency.Builder builder = new BundleDependency.Builder()
        .setDescriptor(bundleDescriptorBuilder.build())
        .setScope(bundleScope);
    // artifact.getFile() may be null if this is a transitive dependency that did not win during a conflict resolution.
    if (!scope.equalsIgnoreCase("provided") && artifact.getFile() != null) {
      builder.setBundleUri(artifact.getFile().toURI());
    }
    return builder;
  }

  static BundleDescriptor.Builder artifactToBundleDescriptor(Artifact artifact) {
    final BundleDescriptor.Builder bundleDescriptorBuilder = new BundleDescriptor.Builder()
        .setArtifactId(artifact.getArtifactId())
        .setGroupId(artifact.getGroupId())
        .setVersion(artifact.getVersion())
        .setBaseVersion(artifact.getBaseVersion())
        .setType(artifact.getExtension());

    String classifier = artifact.getClassifier();
    if (!isEmpty(classifier)) {
      bundleDescriptorBuilder.setClassifier(classifier);
    }

    return bundleDescriptorBuilder;
  }
}
