/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

import java.net.URL;
import java.util.Optional;

public interface RemoteRepository {

  /**
   * Gets the identifier of this repository.
   *
   * @return (case-sensitive) unique identifier, never {@code null}.
   */
  String getId();

  /**
   * @return the remote repository URL
   */
  URL getUrl();

  /**
   * @return authentication credentials for accessing the repository
   */
  Optional<Authentication> getAuthentication();

  /**
   * @return {@link RepositoryPolicy} for releases.
   */
  Optional<RepositoryPolicy> getReleasePolicy();

  /**
   * @return {@link RepositoryPolicy} for snapshots.
   */
  Optional<RepositoryPolicy> getSnapshotPolicy();

  /**
   * @return a {@link RemoteRepositoryBuilder} to build a {@link RemoteRepository}.
   */
  static RemoteRepositoryBuilder newRemoteRepositoryBuilder() {
    return new RemoteRepository.RemoteRepositoryBuilder();
  }

  /**
   * Builder for {@link RemoteRepository} instances
   */
  class RemoteRepositoryBuilder {

    private String id;
    private URL url;
    private Authentication authentication;
    private RepositoryPolicy releasePolicy;
    private RepositoryPolicy snapshotPolicy;

    /**
     * @param id the repository ID, unique and case-sensitive.
     * @return this
     */
    public RemoteRepositoryBuilder id(String id) {
      requireNonNull(id, "id cannot be null");
      this.id = id;
      return this;
    }

    /**
     * @param url the URL to access the remote repository
     * @return this
     */
    public RemoteRepositoryBuilder url(URL url) {
      requireNonNull(url, "remoteRepositoryUrl cannot be null");
      this.url = url;
      return this;
    }

    /**
     * @param authentication authentication credentials to access the remote repository
     * @return this
     */
    public RemoteRepositoryBuilder authentication(Authentication authentication) {
      requireNonNull(authentication, "authentication cannot be null");
      this.authentication = authentication;
      return this;
    }

    /**
     * @param releasePolicy {@link RepositoryPolicy} for release artifacts.
     * @return this
     */
    public RemoteRepositoryBuilder releasePolicy(RepositoryPolicy releasePolicy) {
      requireNonNull(releasePolicy, "releasePolicy cannot be null");
      this.releasePolicy = releasePolicy;
      return this;
    }

    /**
     * @param snapshotPolicy {@link RepositoryPolicy} for snapshot artifacts.
     * @return this
     */
    public RemoteRepositoryBuilder snapshotPolicy(RepositoryPolicy snapshotPolicy) {
      requireNonNull(snapshotPolicy, "snapshotPolicy cannot be null");
      this.snapshotPolicy = snapshotPolicy;
      return this;
    }

    /**
     * Builds the {@link MavenConfiguration} object.
     *
     * @return {@link MavenConfiguration} with the value sets.
     */
    public RemoteRepository build() {
      return new ImmutableRemoteRepository(id, url, ofNullable(authentication), releasePolicy, snapshotPolicy);
    }

  }


}
