/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static java.util.Objects.requireNonNull;

/**
 * Represents the policies for update and checksum on remote repositories.
 */
public class RepositoryPolicy {

  public static final String UPDATE_POLICY_NEVER = "never";
  public static final String UPDATE_POLICY_ALWAYS = "always";
  public static final String UPDATE_POLICY_DAILY = "daily";
  public static final String UPDATE_POLICY_INTERVAL = "interval";
  public static final String CHECKSUM_POLICY_FAIL = "fail";
  public static final String CHECKSUM_POLICY_WARN = "warn";
  public static final String CHECKSUM_POLICY_IGNORE = "ignore";

  private final boolean enabled;
  private final String updatePolicy;
  private final String checksumPolicy;

  private RepositoryPolicy(boolean enabled, String updatePolicy, String checksumPolicy) {
    requireNonNull(updatePolicy, "updatePolicy cannot be null");
    requireNonNull(checksumPolicy, "checksumPolicy cannot be null");
    this.enabled = enabled;
    this.updatePolicy = updatePolicy;
    this.checksumPolicy = checksumPolicy;
  }

  public boolean isEnabled() {
    return this.enabled;
  }

  public String getUpdatePolicy() {
    return this.updatePolicy;
  }

  public String getChecksumPolicy() {
    return this.checksumPolicy;
  }

  public String toString() {
    StringBuilder buffer = new StringBuilder(256);
    buffer.append("enabled=").append(this.isEnabled());
    buffer.append(", checksums=").append(this.getChecksumPolicy());
    buffer.append(", updates=").append(this.getUpdatePolicy());
    return buffer.toString();
  }

  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    } else if (obj != null && this.getClass().equals(obj.getClass())) {
      RepositoryPolicy that = (RepositoryPolicy) obj;
      return this.enabled == that.enabled && this.updatePolicy.equals(that.updatePolicy)
          && this.checksumPolicy.equals(that.checksumPolicy);
    } else {
      return false;
    }
  }

  @Override
  public int hashCode() {
    int result = (enabled ? 1 : 0);
    result = 31 * result + updatePolicy.hashCode();
    result = 31 * result + checksumPolicy.hashCode();
    return result;
  }

  /**
   * @return a {@link RepositoryPolicyBuilder} to build a {@link RepositoryPolicy}.
   */
  public static RepositoryPolicyBuilder newRepositoryPolicyBuilder() {
    return new RepositoryPolicyBuilder();
  }

  /**
   * Builder for {@link RepositoryPolicy} instances
   */
  public static class RepositoryPolicyBuilder {

    private boolean enabled = true;
    private String updatePolicy = UPDATE_POLICY_DAILY;
    private String checksumPolicy = CHECKSUM_POLICY_WARN;

    /**
     * @param enabled enables or disables a remote repository.
     * @return this
     */
    public RepositoryPolicyBuilder enabled(boolean enabled) {
      requireNonNull(enabled, "enabled cannot be null");
      this.enabled = enabled;
      return this;
    }

    /**
     * @param updatePolicy update policy for a remote repository.
     * @return this
     */
    public RepositoryPolicyBuilder updatePolicy(String updatePolicy) {
      requireNonNull(updatePolicy, "updatePolicy cannot be null");
      this.updatePolicy = updatePolicy;
      return this;
    }

    /**
     * @param checksumPolicy checksum policy for a remote repository.
     * @return this
     */
    public RepositoryPolicyBuilder checksumPolicy(String checksumPolicy) {
      requireNonNull(checksumPolicy, "checksumPolicy cannot be null");
      this.checksumPolicy = checksumPolicy;
      return this;
    }

    /**
     * @return a new {@link RepositoryPolicy}.
     */
    public RepositoryPolicy build() {
      return new RepositoryPolicy(enabled, updatePolicy, checksumPolicy);
    }

  }
}
