package org.mule.datasense.impl.util;

import com.google.common.base.Preconditions;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

public class AnnotationSupport<A> {

  private final Map<Class<? extends A>, A> annotationsMap;

  public AnnotationSupport() {
    this.annotationsMap = new LinkedHashMap<>();
  }

  public Set<A> getAnnotations() {
    return new LinkedHashSet<A>(annotationsMap.values());
  }

  public <T extends A> Optional<T> getAnnotation(Class<T> annotationClass) {
    Preconditions.checkNotNull(annotationClass);
    return Optional.ofNullable((T) annotationsMap.get(annotationClass));
  }

  public <T extends A> void annotate(T annotation) {
    Preconditions.checkNotNull(annotation);
    Preconditions.checkState(!annotationsMap.containsKey(annotation), "annotation already present");
    annotationsMap.put((Class<? extends A>) annotation.getClass(), annotation);
  }

  public <T extends A> Optional<T> deannotate(Class<T> annotation) {
    Preconditions.checkNotNull(annotation);
    return Optional.ofNullable((T) annotationsMap.remove(annotation));
  }

  public <T extends A> boolean isAnnotatedWith(Class<T> annotationClass) {
    return annotationsMap.containsKey(annotationClass);
  }

  public <T extends A> T getOrCreateAnnotation(Class<T> annotationClass, Supplier<T> supplier) {
    return getAnnotation(annotationClass).orElseGet(() -> {
      T annotation = supplier.get();
      annotate(annotation);
      return annotation;
    });
  }
}
