package org.mule.datasense.impl.phases.typing.resolver;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeBuilder;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.resolver.errorhandling.ErrorHandlingUtils;
import org.mule.datasense.impl.util.LogSupport;
import org.mule.metadata.MetadataFormats;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;

import com.google.common.base.Throwables;

import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.io.IOUtils;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.api.util.LazyValue;

public class GlobalBindingMetadataTypes implements LogSupport {

  public static final String GLOBAL_BINDING_MULE = "mule";
  public static final String GLOBAL_BINDING_SERVER = "server";
  public static final String GLOBAL_BINDING_APP = "app";
  public static final String GLOBAL_BINDING_FLOW = "flow";
  public static final String GLOBAL_BINDING_CORRELATION_ID = "correlationId";
  public static final String GLOBAL_BINDING_DATA_TYPE = "dataType";
  public static final String GLOBAL_BINDING_AUTHENTICATION = "authentication";
  public static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO = "itemSequenceInfo";
  public static final String GLOBAL_BINDING_ERROR = "error";

  private static final String GLOBAL_BINDING_DATA_TYPE_CLASS = "org.mule.runtime.api.metadata.DataType";
  private static final String GLOBAL_BINDING_AUTHENTICATION_CLASS = "org.mule.runtime.api.security.Authentication";
  private static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS = "org.mule.runtime.api.message.ItemSequenceInfo";
  private static final String ERROR_CLASS = "org.mule.runtime.api.message.Error";
  private static final String TYPED_VALUE_CLASS = "org.mule.runtime.api.metadata.TypedValue";

  public static MetadataType getTypeFromJavaClass(String clazz) {
    return TypesHelper.getTypeFromJavaClass(clazz, GlobalBindingMetadataTypes.class.getClassLoader())
        .orElseThrow(() -> new IllegalArgumentException(format("Failed to resolve class type %s", clazz)));
  }

  private static Optional<MetadataType> getTypeFromWeave(String weaveTypeDef, String typeId, MetadataFormat metadataFormat,
                                                         ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    return expressionLanguageMetadataService
        .createTypeLoader(weaveTypeDef, Optional.ofNullable(metadataFormat).orElse(MetadataFormats.JAVA)).load(typeId);
  }

  private static Optional<MetadataType> getTypeFromWeave(String weaveTypeDef, String typeId,
                                                         ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    return getTypeFromWeave(weaveTypeDef, typeId, null, expressionLanguageMetadataService);
  }


  private static MetadataType metadataTypeFromWeaveResource(String resource, String type,
                                                            ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    try {
      return getTypeFromWeave(IOUtils.toString(GlobalBindingMetadataTypes.class.getResourceAsStream(resource)), type,
                              expressionLanguageMetadataService)
                                  .orElseThrow(() -> new IllegalArgumentException(
                                                                                  format("Failed to resolve weave type %s",
                                                                                         type)));
    } catch (IOException e) {
      throw Throwables.propagate(e);
    }
  }


  private final ExpressionLanguageMetadataService expressionLanguageMetadataService;

  private final MetadataType correlationIdType;
  private final MetadataType dataTypeType;
  private final MetadataType authenticationType;
  private final MetadataType itemSequenceInfoType;
  private final MetadataType typedValueType;
  private final MetadataType appType;
  private final MetadataType serverType;
  private final MetadataType muleType;
  private final MetadataType errorType;
  private final MetadataType flowType;

  public GlobalBindingMetadataTypes(
                                    ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    this.expressionLanguageMetadataService = expressionLanguageMetadataService;

    correlationIdType = CorrelationIdTypeHolder.singleton;
    dataTypeType = DataTypeTypeHolder.singleton;
    authenticationType = AuthenticationTypeHolder.singleton;
    itemSequenceInfoType = ItemSequenceInfoTypeHolder.singleton;
    typedValueType = TypedValueTypeHolder.singleton;
    appType = createAppType();
    serverType = createServerType();
    muleType = createMuleType();
    errorType = ErrorTypeTypeHolder.singleton;
    flowType = createFlowType();
  }

  private static class CorrelationIdTypeHolder {

    private static final MetadataType singleton = getTypeBuilder().stringType().build();

  }

  private static class DataTypeTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_DATA_TYPE_CLASS);

  }

  private static class AuthenticationTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_AUTHENTICATION_CLASS);
  }

  private static class ItemSequenceInfoTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS);
  }

  private static class TypedValueTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(TYPED_VALUE_CLASS);
  }

  private static class ErrorTypeTypeHolder {

    private static final MetadataType singleton =
        TypesHelper.getTypeFromJavaClass(ERROR_CLASS, GlobalBindingMetadataTypes.class.getClassLoader())
            .orElse(ErrorHandlingUtils.errorType(emptySet()));
  }

  public MetadataType correlationIdType() {
    return correlationIdType;
  }

  public MetadataType dataTypeType() {
    return dataTypeType;
  }

  public MetadataType authenticationType() {
    return authenticationType;
  }

  public MetadataType itemSequenceInfoType() {
    return itemSequenceInfoType;
  }

  public MetadataType typedValueType() {
    return typedValueType;
  }

  private static class AppTypeHolder {

    private static AtomicReference<LazyValue<MetadataType>> typeReference = new AtomicReference<>();

    public static MetadataType getValue(ExpressionLanguageMetadataService expressionLanguageMetadataService) {
      typeReference.compareAndSet(null,
                                  new LazyValue<>(() -> metadataTypeFromWeaveResource("app-type.dw", "DataWeaveArtifactContext",
                                                                                      expressionLanguageMetadataService)));
      return typeReference.get().get();
    }
  }

  private MetadataType createAppType() {
    return AppTypeHolder.getValue(expressionLanguageMetadataService);
  }

  public MetadataType appType() {
    return appType;
  }

  private static class ServerTypeHolder {

    private static AtomicReference<LazyValue<MetadataType>> typeReference = new AtomicReference<>();

    public static MetadataType getValue(ExpressionLanguageMetadataService expressionLanguageMetadataService) {
      typeReference.compareAndSet(null, new LazyValue<>(() -> metadataTypeFromWeaveResource("server-type.dw", "ServerContext",
                                                                                            expressionLanguageMetadataService)));
      return typeReference.get().get();
    }
  }

  private MetadataType createServerType() {
    return ServerTypeHolder.getValue(expressionLanguageMetadataService);
  }

  public MetadataType serverType() {
    return serverType;
  }

  private static class MuleTypeHolder {

    private static AtomicReference<LazyValue<MetadataType>> typeReference = new AtomicReference<>();

    public static MetadataType getValue(ExpressionLanguageMetadataService expressionLanguageMetadataService) {
      typeReference.compareAndSet(null, new LazyValue<>(() -> metadataTypeFromWeaveResource("mule-type.dw", "MuleInstanceContext",
                                                                                            expressionLanguageMetadataService)));
      return typeReference.get().get();
    }
  }

  private MetadataType createMuleType() {
    return MuleTypeHolder.getValue(expressionLanguageMetadataService);
  }

  public MetadataType muleType() {
    return muleType;
  }

  public MetadataType errorType() {
    return errorType;
  }

  private static class FlowTypeHolder {

    private static AtomicReference<LazyValue<MetadataType>> typeReference = new AtomicReference<>();

    public static MetadataType getValue(ExpressionLanguageMetadataService expressionLanguageMetadataService) {
      typeReference.compareAndSet(null, new LazyValue<>(() -> metadataTypeFromWeaveResource("flow-type.dw", "NamedObject",
                                                                                            expressionLanguageMetadataService)));
      return typeReference.get().get();
    }
  }

  private MetadataType createFlowType() {
    return FlowTypeHolder.getValue(expressionLanguageMetadataService);
  }

  public MetadataType flowType() {
    return flowType;
  }

}
