package org.mule.datasense.impl.phases.builder;

import org.mule.datasenseapi.api.ApplicationResolutionScope;
import org.mule.datasenseapi.api.ComponentResolutionScope;
import org.mule.datasenseapi.api.ConfigResolutionScope;
import org.mule.datasenseapi.api.DataSenseResolutionScope;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolver;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.util.TreeLogSupport;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.Optional;

public class MuleAstParserContext {

  private final TreeLogSupport treeLogSupport;
  private final ComponentModelClassifier componentModelClassifier;
  private final DataSenseResolutionScopeStrategy dataSenseResolutionScopeStrategy;
  private final TypeResolverRegistry typeResolverRegistry;

  public MuleAstParserContext(ApplicationModel applicationModel, DataSenseProvider dataSenseProvider,
                              ComponentModelClassifier componentModelClassifier) {
    this(applicationModel, dataSenseProvider, componentModelClassifier, null, null);
  }

  public MuleAstParserContext(ApplicationModel applicationModel, DataSenseProvider dataSenseProvider,
                              ComponentModelClassifier componentModelClassifier,
                              DataSenseResolutionScope dataSenseResolutionScope, TypeResolverRegistry typeResolverRegistry) {
    this.componentModelClassifier = componentModelClassifier;
    this.dataSenseResolutionScopeStrategy =
        createDataSenseResolutionScopeStrategy(applicationModel, dataSenseProvider, dataSenseResolutionScope);
    this.typeResolverRegistry = typeResolverRegistry;
    treeLogSupport = new TreeLogSupport();
  }

  public Optional<DataSenseResolutionScopeStrategy> getDataSenseResolutionScopeStrategy() {
    return Optional.ofNullable(dataSenseResolutionScopeStrategy);
  }

  TreeLogSupport astlogger() {
    return treeLogSupport;
  }

  void enter(ComponentAst componentModel) {
    astlogger().enter(componentModel.getIdentifier().toString());
  }

  void exit(ComponentAst componentModel) {
    astlogger().exit(componentModel.getIdentifier().toString());
  }

  public Optional<ComponentModelType> getComponentModelType(ComponentAst componentModel) {
    return componentModelClassifier
        .getComponentModelType(componentModel);
  }

  public Optional<MuleAstParseProvider> getParseProvider(
                                                         ComponentAst componentModel) {
    return typeResolverRegistry.get(componentModel.getIdentifier()).flatMap(TypeResolver::getParseProvider);
  }

  private static DataSenseResolutionScopeStrategy createDataSenseResolutionScopeStrategy(ApplicationModel applicationModel,
                                                                                         DataSenseProvider dataSenseProvider,
                                                                                         DataSenseResolutionScope dataSenseResolutionScope) {
    if (dataSenseResolutionScope == null) {
      return null;
    }

    if (dataSenseResolutionScope instanceof ApplicationResolutionScope) {
      return new ApplicationResolutionScopeStrategy((ApplicationResolutionScope) dataSenseResolutionScope);
    } else if (dataSenseResolutionScope instanceof ConfigResolutionScope) {
      return new SimpleDependenciesResolutionScopeStrategy(applicationModel, dataSenseProvider,
                                                           new ConfigResolutionScopeStrategy((ConfigResolutionScope) dataSenseResolutionScope));
    } else if (dataSenseResolutionScope instanceof ComponentResolutionScope) {
      return new SimpleDependenciesResolutionScopeStrategy(applicationModel, dataSenseProvider,
                                                           new ComponentResolutionScopeStrategy((ComponentResolutionScope) dataSenseResolutionScope));
    } else {
      throw new IllegalArgumentException(String.format("Unknown resolution scope class %s.",
                                                       dataSenseResolutionScope.getClass()));
    }
  }
}
