package org.mule.datasense.impl.model.ast;

import org.mule.datasenseapi.api.notifications.DataSenseNotification;
import org.mule.datasenseapi.api.notifications.DataSenseNotificationType;
import org.mule.datasense.impl.util.LocationUtils;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.runtime.api.metadata.resolving.FailureCode;
import org.mule.runtime.api.metadata.resolving.MetadataComponent;
import org.mule.runtime.api.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class AstNotification {

  public static final transient Logger logger = LoggerFactory.getLogger("org.mule.datasense.notifications");


  private class AstNotificationEntry implements DataSenseNotification {

    private final DataSenseNotificationType notificationType;
    private AstNodeLocation astNodeLocation;
    private I18nMessage message;
    private MetadataComponent metadataComponent;
    private String failingElement;
    private FailureCode failureCode;
    private I18nMessage reason;

    public AstNotificationEntry(DataSenseNotificationType notificationType, AstNodeLocation astNodeLocation, I18nMessage message,
                                FailureCode failureCode, MetadataComponent metadataComponent, String failingElement,
                                I18nMessage reason) {
      Preconditions.checkNotNull(notificationType, "NotificationType not present.");
      Preconditions.checkNotNull(message, "Message not present.");
      this.notificationType = notificationType;
      this.astNodeLocation = astNodeLocation;
      this.message = message;
      this.failureCode = failureCode;
      this.metadataComponent = metadataComponent;
      this.failingElement = failingElement;
      this.reason = reason;
    }

    public AstNotificationEntry(DataSenseNotificationType notificationType, AstNodeLocation astNodeLocation,
                                I18nMessage message) {
      this(notificationType, astNodeLocation, message, null, null, null, null);
    }

    public AstNotificationEntry(DataSenseNotificationType notificationType, I18nMessage message) {
      this(notificationType, null, message);
    }

    public DataSenseNotificationType getNotificationType() {
      return notificationType;
    }

    @Override
    public I18nMessage getMessage() {
      return message;
    }

    public Optional<AstNodeLocation> getAstNodeLocation() {
      return Optional.ofNullable(astNodeLocation);
    }

    @Override
    public Optional<ComponentLocation> getComponentLocation() {
      return getAstNodeLocation().map(AstNodeLocation::getComponentLocation);
    }

    @Override
    public Optional<MetadataComponent> getMetadataComponent() {
      return Optional.ofNullable(metadataComponent);
    }

    @Override
    public Optional<String> getFailingElement() {
      return Optional.ofNullable(failingElement);
    }

    @Override
    public Optional<FailureCode> getFailureCode() {
      return Optional.ofNullable(failureCode);
    }

    @Override
    public Optional<I18nMessage> getReason() {
      return Optional.ofNullable(reason);
    }

    @Override
    public String toString() {
      return "AstNotificationEntry{" +
          "notificationType=" + notificationType +
          ", astNodeLocation=" + astNodeLocation +
          ", message=" + message +
          ", metadataComponent=" + metadataComponent +
          ", failingElement='" + failingElement + '\'' +
          ", failureCode=" + failureCode +
          ", reason=" + reason +
          '}';
    }
  }

  private final List<AstNotificationEntry> notifications;

  public AstNotification() {
    notifications = new LinkedList<>();
  }

  public List<String> getPlainMessages() {
    return notifications.stream().map(this::plainMessage).collect(Collectors.toList());
  }

  public List<DataSenseNotification> getDataSenseNotifications() {
    return getDataSenseNotifications(dataSenseNotification -> true);
  }

  public List<DataSenseNotification> getDataSenseNotifications(Predicate<DataSenseNotification> dataSenseNotificationPredicate) {
    return notifications.stream().filter(dataSenseNotificationPredicate).collect(Collectors.toList());
  }

  public List<DataSenseNotification> getDataSenseNotifications(Location location) {
    return getDataSenseNotifications(dataSenseNotification -> location.equals(dataSenseNotification.getComponentLocation()
        .map(componentLocation -> LocationUtils.valueOf(componentLocation.getLocation()))
        .orElse(null)));
  }

  private String plainMessage(AstNotificationEntry astNotificationEntry) {
    return String.format("[%s] %s: Message: %s. %s", astNotificationEntry.getNotificationType(),
                         astNotificationEntry.getAstNodeLocation().map(
                                                                       AstNodeLocation::toString)
                             .orElse("unknown"),
                         astNotificationEntry.getMessage().getMessage(),
                         astNotificationEntry.getReason().map(i18nMessage -> " - Reason: " + i18nMessage.getMessage())
                             .orElse(""));
  }

  private void log(AstNotificationEntry astNotificationEntry) {
    /*
    switch (astNotificationEntry.getNotificationType()) {
      case FATAL_ERROR:
      case ERROR:
        if (logger.isErrorEnabled()) {
          logger.error(plainMessage(astNotificationEntry));
        }
        break;
      case WARNING:
        if (logger.isWarnEnabled()) {
          logger.warn(plainMessage(astNotificationEntry));
        }
        break;
      case INFO:
      default:
        if (logger.isInfoEnabled()) {
          logger.info(plainMessage(astNotificationEntry));
        }
        break;
    }
    */
    logger.debug(plainMessage(astNotificationEntry));
  }

  private void report(AstNotificationEntry astNotificationEntry) {
    log(astNotificationEntry);
    notifications.add(astNotificationEntry);
  }

  public void reportFatalError(I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.FATAL_ERROR, message));
  }

  public void reportFatalError(AstNodeLocation astNodeLocation, I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.FATAL_ERROR, astNodeLocation, message));
  }

  public void reportFatalError(AstNodeLocation astNodeLocation, I18nMessage message, FailureCode failureCode,
                               MetadataComponent metadataComponent, String failingElement, I18nMessage reason) {
    report(new AstNotificationEntry(DataSenseNotificationType.FATAL_ERROR, astNodeLocation, message, failureCode,
                                    metadataComponent, failingElement, reason));
  }

  public void reportError(AstNodeLocation astNodeLocation, I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.ERROR, astNodeLocation, message));
  }

  public void reportError(I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.ERROR, null, message));
  }

  public void reportError(AstNodeLocation astNodeLocation, I18nMessage message, FailureCode failureCode,
                          MetadataComponent metadataComponent, String failingElement, I18nMessage reason) {
    report(new AstNotificationEntry(DataSenseNotificationType.ERROR, astNodeLocation, message, failureCode, metadataComponent,
                                    failingElement, reason));
  }

  public void reportWarning(AstNodeLocation astNodeLocation, I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.WARNING, astNodeLocation, message));
  }

  public void reportWarning(I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.WARNING, message));
  }

  public void reportWarning(AstNodeLocation astNodeLocation, I18nMessage message, FailureCode failureCode,
                            MetadataComponent metadataComponent, String failingElement, I18nMessage reason) {
    report(new AstNotificationEntry(DataSenseNotificationType.WARNING, astNodeLocation, message, failureCode, metadataComponent,
                                    failingElement, reason));
  }

  public void reportInfo(AstNodeLocation astNodeLocation, I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.INFO, astNodeLocation, message));
  }

  public void reportInfo(I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.INFO, message));
  }

  public void reportDebug(I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.DEBUG, message));
  }

  public void reportDebug(AstNodeLocation astNodeLocation, I18nMessage message) {
    report(new AstNotificationEntry(DataSenseNotificationType.DEBUG, astNodeLocation, message));
  }

  private boolean hasNotificationWithType(DataSenseNotificationType notificationType) {
    return notifications.stream().filter(astNotificationEntry -> astNotificationEntry.notificationType == notificationType)
        .findFirst().isPresent();
  }

  public boolean hasErrors() {
    return hasNotificationWithType(DataSenseNotificationType.ERROR);
  }

  public boolean hasWarnings() {
    return hasNotificationWithType(DataSenseNotificationType.WARNING);
  }

  public boolean hasFatalErrors() {
    return hasNotificationWithType(DataSenseNotificationType.FATAL_ERROR);
  }
}
