package org.mule.datasense.impl.phases.annotators;

import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.operation.OperationCallBuilder;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.util.extension.DslElementModelVisitor;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public abstract class StaticDslBaseAnnotator extends BaseOperationCallBuilderAnnotator {

  protected static class OperationCallDslElementModelVisitor implements DslElementModelVisitor {

    private final OperationCallBuilder operationCallBuilder;
    private boolean hasDynamicType;
    private List<ParameterModel> incompleteForDynamicMetadata;

    public OperationCallDslElementModelVisitor(OperationCallBuilder operationCallBuilder) {
      this.operationCallBuilder = operationCallBuilder;
      hasDynamicType = false;
      incompleteForDynamicMetadata = new ArrayList<>();
    }

    @Override
    public void visitOperationModel(ComponentAst operationAst) {
      final OperationModel model = operationAst.getModel(OperationModel.class).get();
      operationCallBuilder.name(model.getName());

      operationAst.getParameters().stream()
          .forEach(paramAst -> onParameter(paramAst.getModel(), paramAst.getValue()));

      operationCallBuilder.errors(model.getErrorModels());
      operationCallBuilder.returnType(TypesHelper.getMessageMetadataTypeBuilder()
          .payload(model.getOutput().getType())
          .attributes(model.getOutputAttributes().getType()).build());
      hasDynamicType |= model.getOutput().hasDynamicType() || model.getOutputAttributes().hasDynamicType();
      if (hasDynamicType) {
        incompleteForDynamicMetadata = incompleteForDynamicMetadata(operationCallBuilder, model);
      }
    }

    private static List<ParameterModel> incompleteForDynamicMetadata(OperationCallBuilder operationCallBuilder,
                                                                     ParameterizedModel parameterizedModel) {
      List<ParameterModel> incompleteParameters = new ArrayList<>();

      if (operationCallBuilder != null) {
        final OperationCall operationCall = operationCallBuilder.build();
        final List<ParameterModel> allParameterModels = parameterizedModel.getAllParameterModels();
        if (allParameterModels != null) {
          for (ParameterModel parameterModel : allParameterModels) {
            final MetadataKeyPartModelProperty metadataKeyPartModelProperty =
                parameterModel.getModelProperty(MetadataKeyPartModelProperty.class).orElse(null);
            if (parameterModel.isRequired() && metadataKeyPartModelProperty != null) {
              if (!operationCall.getInputMapping(parameterModel.getName()).isPresent()) {
                incompleteParameters.add(parameterModel);
              } ;
            }
          }
        }
      }

      return incompleteParameters;
    }

    private List<ParameterModel> getSourceDeclaredParameterModels(SourceModel sourceModel) {
      if (sourceModel == null) {
        return Collections.emptyList();
      }

      return sourceModel.getParameterGroupModels().stream().flatMap((parameterGroupModel) -> {
        return parameterGroupModel.getParameterModels().stream();
      }).collect(Collectors.toList());
    }

    @Override
    public void visitSourceModel(ComponentAst sourceAst) {
      final SourceModel model = sourceAst.getModel(SourceModel.class).get();
      operationCallBuilder.name(model.getName());

      final List<ParameterModel> sourceDeclaredParameterModels = getSourceDeclaredParameterModels(model);

      sourceAst.getParameters().stream()
          .filter(componentParameterAst -> sourceDeclaredParameterModels.contains(componentParameterAst.getModel()))
          .forEach(paramAst -> onParameter(paramAst.getModel(), paramAst.getValue()));

      operationCallBuilder.errors(model.getErrorModels());
      operationCallBuilder.returnType(TypesHelper.getMessageMetadataTypeBuilder()
          .payload(model.getOutput().getType())
          .attributes(model.getOutputAttributes().getType()).build());
      hasDynamicType |= model.getOutput().hasDynamicType() || model.getOutputAttributes().hasDynamicType();
      incompleteForDynamicMetadata = incompleteForDynamicMetadata(operationCallBuilder, model);
    }

    private void addParameter(String name, MetadataType metadataType, Either<String, Object> value) {
      operationCallBuilder.input(inputMappingBuilder -> inputMappingBuilder
          .parameter(inputParameterBuilder -> inputParameterBuilder
              .name(name)
              .type(metadataType))
          .argument(inputArgumentBuilder -> {
            value.applyLeft(left -> {
              inputArgumentBuilder.expression(left);
              inputArgumentBuilder.value(left);
            });
            value.applyRight(right -> {
              if (right instanceof String) {
                inputArgumentBuilder.value((String) right);
              }
            });
          }));
    }

    private void onParameter(final ParameterModel parameterModel, Either<String, Object> value) {
      hasDynamicType |= parameterModel.hasDynamicType();
      value.getValue().ifPresent(v -> addParameter(parameterModel.getName(), parameterModel.getType(), value));
    }

    public boolean isHasDynamicType() {
      return hasDynamicType;
    }

    public Collection<ParameterModel> isIncompleteForDynamicMetadata() {
      return incompleteForDynamicMetadata;
    }
  }
}
