package org.mule.datasense.impl.model.annotations;

import static java.util.Optional.ofNullable;

import org.mule.datasense.api.metadataprovider.DataSenseConfiguration;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataProvider;
import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.util.LogSupport;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.EnumLabelsAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.StringTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

public abstract class ComponentModelMetadataKeyEnricher<T> implements LogSupport {

  protected String getPartName(Set<MetadataKey> metadataKeys) {
    if (metadataKeys == null) {
      return null;
    }

    return metadataKeys.stream().findFirst().map(MetadataKey::getPartName).orElse(null);
  }

  protected String[] enumOfKeyValues(Set<MetadataKey> metadataKeys) {
    final List<String> collect = metadataKeys.stream().map(MetadataKey::getId).collect(Collectors.toList());
    return collect.toArray(new String[collect.size()]);
  }

  protected Optional<String> findKeyPartConfiguredValue(ComponentAst componentAst, String keyPartName) {
    ComponentParameterAst parameter = componentAst.getParameters()
        .stream()
        .filter(p -> p.getModel().getName().equals(keyPartName))
        .findFirst()
        .get();


    return parameter.isDefaultValue() ? Optional.empty()
        : ofNullable(parameter.getValue()).filter(value -> value.isRight() && value.getRight() instanceof String)
            .map(value -> (String) value.getRight());
  }

  protected void filterMetadataKeys(Map<String, Set<MetadataKey>> metadataKeysByPart, Set<MetadataKey> metadataKeys,
                                    ComponentAst componentAst) {
    final Optional<String> optionalKeyPart = metadataKeys.stream().findFirst().map(MetadataKey::getPartName);
    if (!optionalKeyPart.isPresent()) {
      return;
    }
    String keyPart = optionalKeyPart.get();

    metadataKeysByPart.put(keyPart, metadataKeys);

    final Optional<String> optionalKeyPartConfiguredValue = findKeyPartConfiguredValue(componentAst, keyPart);
    if (!optionalKeyPartConfiguredValue.isPresent()) {
      return;
    } else {
      final String keyPartConfiguredValue = optionalKeyPartConfiguredValue.get();

      final Optional<MetadataKey> matchingKey =
          metadataKeys.stream().filter(metadataKey -> metadataKey.getId().equals(keyPartConfiguredValue)).findFirst();

      if (matchingKey.isPresent()) {
        filterMetadataKeys(metadataKeysByPart, matchingKey.get().getChilds(), componentAst);
      } else {
        metadataKeysByPart.put(keyPart, metadataKeys);
        return;
      }
    }
  }

  protected Function<ParameterModel, MetadataType> getParameterModelMetadataTypeFunction(Map<String, Set<MetadataKey>> metadataKeysByPart,
                                                                                         OperationCall operationCall) {
    Map<String, MetadataType> operationCallParameters = new HashMap<>();
    if (operationCall != null) {
      operationCall.getInputMappings().forEach(inputMapping -> {
        final String name = inputMapping.getInputParameter().getName();
        final MetadataType metadataType = inputMapping.getInputParameter().getMetadataType();
        if (name != null && metadataType != null) {
          operationCallParameters.put(name, metadataType);
        }
      });
    }
    return parameterModel -> {
      final Optional<Set<MetadataKey>> metadataKeys = Optional.ofNullable(metadataKeysByPart.get(parameterModel.getName()));
      if (metadataKeys.isPresent() && parameterModel.getType() instanceof StringType) {
        final StringTypeBuilder typeBuilder = TypesHelper.getTypeBuilder(MetadataFormat.JAVA).stringType();
        final Set<TypeAnnotation> annotations = parameterModel.getType().getAnnotations();
        enrichAnnotations(metadataKeys.get(), typeBuilder, annotations);
        return typeBuilder.build();
      } else if (operationCallParameters.containsKey(parameterModel.getName())) {
        return operationCallParameters.get(parameterModel.getName());
      } else {
        return parameterModel.getType();
      }
    };
  }

  private void enrichAnnotations(Set<MetadataKey> metadataKeys, StringTypeBuilder typeBuilder,
                                 Set<TypeAnnotation> annotations) {
    Set<TypeAnnotation> newAnnotations = new HashSet<>();
    annotations.stream().filter(typeAnnotation -> !(typeAnnotation instanceof EnumAnnotation
        || typeAnnotation instanceof EnumLabelsAnnotation)).forEach(newAnnotations::add);
    List<String> values = new ArrayList<>();
    List<String> labels = new ArrayList<>();
    metadataKeys.forEach(metadataKey -> {
      values.add(metadataKey.getId());
      labels.add(metadataKey.getDisplayName());
    });
    newAnnotations.add(new EnumAnnotation<>(values.toArray(new String[values.size()])));
    newAnnotations.add(new EnumLabelsAnnotation(labels.toArray(new String[labels.size()])));
    newAnnotations.forEach(typeBuilder::with);
  }

  private Optional<MessageMetadataType> returnTypeAsMessageMetadatatype(OperationCall operationCall) {
    if (operationCall == null) {
      return Optional.empty();
    }

    MetadataType metadataType = operationCall.getReturnType();
    return Optional.ofNullable(metadataType instanceof MessageMetadataType ? (MessageMetadataType) metadataType : null);
  }

  protected Function<OutputModel, MetadataType> getOutputMetadataTypeFunction(OperationCall operationCall) {
    return outputModel -> returnTypeAsMessageMetadatatype(operationCall).flatMap(
                                                                                 MessageMetadataType::getPayloadType)
        .orElse(outputModel.getType());
  }

  protected Function<OutputModel, MetadataType> getOutputAttributesMetadataTypeFunction(OperationCall operationCall) {
    return outputModel -> returnTypeAsMessageMetadatatype(operationCall).flatMap(
                                                                                 MessageMetadataType::getAttributesType)
        .orElse(outputModel.getType());
  }

  public T enrich(T model, ComponentAst componentAst,
                  OperationCall operationCall, Location location,
                  DataSenseMetadataProvider dataSenseMetadataProvider, AstNotification astNotification,
                  Optional<DataSenseConfiguration> dataSenseConfigurationOptional) {
    T result;

    Map<String, Set<MetadataKey>> metadataKeysByPart = new HashMap<>();

    if (dataSenseConfigurationOptional.map(dataSenseConfiguration -> dataSenseConfiguration.isKeyEnrichment()).orElse(true)
        && hasKeyParameter(componentAst)) {
      final MetadataResult<MetadataKeysContainer> metadataResult =
          dataSenseMetadataProvider.getMetadataKeys(componentAst);
      final ComponentLocation componentLocation = componentAst.getLocation();
      if (metadataResult != null) {
        if (metadataResult.isSuccess()) {
          final MetadataKeysContainer metadataKeysContainer = metadataResult.get();
          // metadatakeys resolved for an operation have unique category
          metadataKeysContainer.getKeysByCategory().values().stream().findFirst()
              .ifPresent(metadataKeys -> filterMetadataKeys(metadataKeysByPart, metadataKeys, componentAst));
        } else {
          metadataResult.getFailures().forEach(metadataFailure -> {
            astNotification.reportError(new AstNodeLocation(componentLocation),
                                        NotificationMessages.MSG_FAILED_TO_ENRICH_KEYS(metadataFailure),
                                        metadataFailure.getFailureCode(),
                                        metadataFailure.getFailingComponent(),
                                        metadataFailure.getFailingElement().orElse(null),
                                        NotificationMessages.REASON_FAILED_TO_ENRICH_KEYS(metadataFailure));
          });
        }
      } else {
        astNotification
            .reportError(new AstNodeLocation(componentLocation), I18nMessageFactory
                .createStaticMessage(NotificationMessages.MSG_FAILED_TO_OBTAIN_METADATA,
                                     "metadatakeys for component_path: " + location));
      }
    }

    result = enrich(model, metadataKeysByPart, operationCall);

    return result;
  }

  private boolean hasKeyParameter(ComponentAst componentAst) {
    return componentAst.getModel(ComponentModel.class)
        .map(model -> model.getAllParameterModels().stream()
            .anyMatch(parameterModel -> parameterModel.getModelProperty(MetadataKeyPartModelProperty.class).isPresent()))
        .orElse(false);
  }

  protected abstract T enrich(T componentModel, Map<String, Set<MetadataKey>> metadataKeysByPart, OperationCall operationCall);


}
