package org.mule.datasense.types.parser;

import java.util.Arrays;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Parsers {

  public static Parser<?> empty() {
    return success(unit);
  }

  public static <B> Parser<B> success(B value) {
    return in -> value;
  }

  public static <B> Parser<B> failure(String cause) {
    return in -> {
      throw Failure.exception;
    };
  }

  @SafeVarargs
  public static <B> Parser<B> choice(Parser<B> parser, Parser<B>... parsers) {
    return Arrays.asList(parsers).stream().reduce(parser, Parser::or);
  }

  public static SkipParser skip(Parser<?> parser) {
    return new SkipParser(parser);
  }

  public static <B> Parser<B> then(Supplier<Parser<B>> parser) {
    return in -> parser.get().parse(in);
  }

  public static Parser<?> begin() {
    return regex("^").map(m -> unit);
  }

  public static Parser<?> end() {
    return regex("$").map(m -> unit);
  }

  public static Parser<String> string(String string) {
    return regex(Pattern.quote(string)).map(MatchResult::group);
  }

  public static Parser<MatchResult> regex(String regex) {
    return regex(Pattern.compile(regex));
  }

  public static Parser<MatchResult> regex(Pattern pattern) {
    return new RegexParser(pattern);
  }

  public static class Failure extends Exception {

    public static final Failure exception = new Failure();
  }


  public static class Input {

    public Input(CharSequence charSequence, int offset) {
      this.charSequence = charSequence;
      this.offset = offset;
    }

    CharSequence charSequence;
    int offset;
  }


  public static class Pair<X, Y> {

    public final X first;
    public final Y second;

    public Pair(X first, Y second) {
      this.first = first;
      this.second = second;
    }

    @Override
    public String toString() {
      return "(" + first + ", " + second + ")";
    }
  }

  // Functions to match on left-nested pairs, like those generated by a chain of .then(...) calls

  /**
   * Match a single value, like the result of string("A") - note that match is mostly useful when dealing with multiple values.
   */
  public static <A, R> Function<A, R> match(Function<A, R> f) {
    return f;
  }

  /**
   * Match two values, like the result of string("A").then(string("B"))
   */
  public static <A, B, R> Function<Pair<A, B>, R> match(BiFunction<A, B, R> f) {
    return p -> f.apply(p.first, p.second);
  }

  /**
   * Match three values, like the result of string("A").then(string("B")).then(string("C"))
   */
  public static <A, B, C, R> Function<Pair<Pair<A, B>, C>, R> match(Function3<A, B, C, R> f) {
    return p -> f.apply(p.first.first, p.first.second, p.second);
  }

  /**
   * Match four values, like the result of string("A").then(string("B")).then(string("C")).then(string("D"))
   */
  public static <A, B, C, D, R> Function<Pair<Pair<Pair<A, B>, C>, D>, R> match(Function4<A, B, C, D, R> f) {
    return p -> f.apply(p.first.first.first, p.first.first.second, p.first.second, p.second);
  }

  /**
   * Match five values, like the result of string("A").then(string("B")).then(string("C")).then(string("D")).then(string("E"))
   */
  public static <A, B, C, D, E, R> Function<Pair<Pair<Pair<Pair<A, B>, C>, D>, E>, R> match(Function5<A, B, C, D, E, R> f) {
    return p -> f
        .apply(p.first.first.first.first, p.first.first.first.second, p.first.first.second, p.first.second, p.second);
  }

  /**
   * Match six values, like the result of string("A").then(string("B")).then(string("C"))...then(string("F"))
   */
  public static <A, B, C, D, E, F, R> Function<Pair<Pair<Pair<Pair<Pair<A, B>, C>, D>, E>, F>, R> match(
                                                                                                        Function6<A, B, C, D, E, F, R> f) {
    return p -> f.apply(p.first.first.first.first.first, p.first.first.first.first.second, p.first.first.first.second,
                        p.first.first.second, p.first.second, p.second);
  }

  /**
   * Match seven values, like the result of string("A").then(string("B")).then(string("C"))...then(string("G"))
   */
  public static <A, B, C, D, E, F, G, R> Function<Pair<Pair<Pair<Pair<Pair<Pair<A, B>, C>, D>, E>, F>, G>, R> match(
                                                                                                                    Function7<A, B, C, D, E, F, G, R> f) {
    return p -> f.apply(p.first.first.first.first.first.first, p.first.first.first.first.first.second,
                        p.first.first.first.first.second, p.first.first.first.second, p.first.first.second, p.first.second,
                        p.second);
  }

  /**
   * Match eight values, like the result of string("A").then(string("B")).then(string("C"))...then(string("H"))
   */
  public static <A, B, C, D, E, F, G, H, R> Function<Pair<Pair<Pair<Pair<Pair<Pair<Pair<A, B>, C>, D>, E>, F>, G>, H>, R> match(
                                                                                                                                Function8<A, B, C, D, E, F, G, H, R> f) {
    return p -> f.apply(p.first.first.first.first.first.first.first, p.first.first.first.first.first.first.second,
                        p.first.first.first.first.first.second, p.first.first.first.first.second, p.first.first.first.second,
                        p.first.first.second, p.first.second, p.second);
  }

  /**
   * Match nine values, like the result of string("A").then(string("B")).then(string("C"))...then(string("I"))
   */
  public static <A, B, C, D, E, F, G, H, I, R> Function<Pair<Pair<Pair<Pair<Pair<Pair<Pair<Pair<A, B>, C>, D>, E>, F>, G>, H>, I>, R> match(
                                                                                                                                            Function9<A, B, C, D, E, F, G, H, I, R> f) {
    return p -> f.apply(p.first.first.first.first.first.first.first.first, p.first.first.first.first.first.first.first.second,
                        p.first.first.first.first.first.first.second, p.first.first.first.first.first.second,
                        p.first.first.first.first.second, p.first.first.first.second, p.first.first.second, p.first.second,
                        p.second);
  }

  @FunctionalInterface
  public interface Function3<A, B, C, R> {

    public R apply(A a, B b, C c);
  }


  @FunctionalInterface
  public interface Function4<A, B, C, D, R> {

    public R apply(A a, B b, C c, D d);
  }


  @FunctionalInterface
  public interface Function5<A, B, C, D, E, R> {

    public R apply(A a, B b, C c, D d, E e);
  }


  @FunctionalInterface
  public interface Function6<A, B, C, D, E, F, R> {

    public R apply(A a, B b, C c, D d, E e, F f);
  }


  @FunctionalInterface
  public interface Function7<A, B, C, D, E, F, G, R> {

    public R apply(A a, B b, C c, D d, E e, F f, G g);
  }


  @FunctionalInterface
  public interface Function8<A, B, C, D, E, F, G, H, R> {

    public R apply(A a, B b, C c, D d, E e, F f, G g,
                   H h);
  }


  @FunctionalInterface
  public interface Function9<A, B, C, D, E, F, G, H, I, R> {

    public R apply(A a, B b, C c, D d, E e, F f,
                   G g, H h, I i);
  }


  // A value to use when no result is needed from a parser (eg. for Parser<?>)
  public static Object unit = new Object();

  // Package private stuff


  public static class SkipParser {

    private final Parser<?> parser;

    public SkipParser(Parser<?> parser) {
      this.parser = parser;
    }

    public SkipParser skip(Parser<?> that) {
      return new SkipParser(in -> {
        parser.parse(in);
        that.parse(in);
        return Parsers.unit;
      });
    }

    public <U> Parser<U> then(Parser<U> that) {
      return in -> {
        parser.parse(in);
        return that.parse(in);
      };
    }

    public <B> Parser<B> then(Supplier<Parser<B>> that) {
      return in -> {
        parser.parse(in);
        return that.get().parse(in);
      };
    }
  }


  public static class RegexParser implements Parser<MatchResult> {

    private Pattern pattern;

    public RegexParser(Pattern pattern) {
      this.pattern = pattern;
    }

    public MatchResult parse(Input in) throws Failure {
      Matcher matcher = pattern.matcher(in.charSequence.subSequence(in.offset, in.charSequence.length()));
      if (matcher.lookingAt()) {
        in.offset += matcher.end();
        return matcher.toMatchResult();
      } else {
        throw Failure.exception;
      }
    }
  }

}
