package org.mule.datasense.common.loader.json;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;

import java.util.Optional;
import java.util.stream.IntStream;
import java.util.stream.Stream;

public class JsonMatcher {

  static Stream<JsonElement> jsonArrayStream(JsonArray jsonArray) {
    return IntStream.range(0, jsonArray.size()).mapToObj(jsonArray::get);
  }

  public static Optional<JsonMatcher> createMatcher(JsonElement jsonElement) {
    if (jsonElement == null) {
      return Optional.empty();
    } else {
      return Optional.of(new JsonMatcher(createStrategy(jsonElement)));
    }
  }

  public static Optional<JsonMatcher> match(JsonElement jsonElement, String name) {
    return createMatcher(jsonElement).flatMap(jsonMatcher -> jsonMatcher.match(name));
  }

  static JsonMatcher createRequiredMatcher(JsonElement jsonElement) {
    return createMatcher(jsonElement).orElseThrow(IllegalArgumentException::new);
  }


  static JsonMatcherStrategy createStrategy(JsonElement jsonElement) {
    if (jsonElement instanceof JsonObject) {
      return new JsonObjectStrategy((JsonObject) jsonElement);
    } else if (jsonElement instanceof JsonArray) {
      return new JsonArrayStrategy((JsonArray) jsonElement);
    } else {
      throw new IllegalArgumentException();
    }
  }

  private JsonMatcherStrategy jsonMatcherStrategy;

  private JsonMatcher(JsonMatcherStrategy jsonMatcherStrategy) {
    this.jsonMatcherStrategy = jsonMatcherStrategy;
  }

  public Optional<JsonMatcher> match(String name) {
    return jsonMatcherStrategy.match(name);
  }

  public JsonMatcher require(String name) {
    return jsonMatcherStrategy.match(name)
        .orElseThrow(() -> new JsonMatcherException(String.format("Missing required '%s'", name)));
  }

  public Optional<Stream<JsonMatcher>> matchMany(String name) {
    return jsonMatcherStrategy.matchMany(name);
  }

  public Stream<JsonMatcher> requireMany(String name) {
    return jsonMatcherStrategy.matchMany(name)
        .orElseThrow(() -> new JsonMatcherException(String.format("Missing required many '%s'", name)));
  }

  public Optional<Stream<JsonMatcher>> matchMany() {
    return jsonMatcherStrategy.matchMany();
  }

  public Optional<JsonPrimitive> matchAttribute(String name) {
    return jsonMatcherStrategy.matchAttribute(name);
  }

  public Optional<String> matchString(String name) {
    return jsonMatcherStrategy.matchAttribute(name).map(jsonPrimitive -> jsonPrimitive.getAsString());
  }

  public JsonPrimitive requireAttribute(String name) {
    return matchAttribute(name).orElseThrow(() -> new JsonMatcherException(String.format("Missing attribute '%s'", name)));
  }

  public String requireString(String name) {
    return requireAttribute(name).getAsString();
  }

  public JsonElement element() {
    return jsonMatcherStrategy.element();
  }
}
