package org.mule.datasense.declarations.loader.xml;

import com.google.common.base.Throwables;
import org.mule.datasense.catalog.model.resolver.TypeResolverException;
import org.mule.datasense.common.loader.xml.XmlMatcher;
import org.mule.datasense.common.util.Notifier;
import org.mule.datasense.common.util.TypeUtils;
import org.mule.datasense.declarations.loader.AbstractTypeDeclarationLoader;
import org.mule.datasense.declarations.loader.TypeDeclarationLoaderContext;
import org.mule.datasense.declarations.model.MessageProcessorTypeDeclaration;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.FunctionTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.w3c.dom.Element;

import javax.xml.namespace.QName;
import java.util.Optional;

import static org.mule.datasense.declarations.util.NotificationMessages.MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION;

public class MessageProcessorTypeDeclarationXmlLoader
    extends AbstractTypeDeclarationLoader<MessageProcessorTypeDeclaration, Element> {

  private static final String NS_MULE = "http://www.mulesoft.org/schema/mule/types";
  private static final String NS_TYPES = "http://www.mulesoft.org/schema/mule/types";
  private static final QName ELEM_MP_DECLARATION = new QName(NS_TYPES, "processor-declaration");
  private static final QName ELEM_INPUT_EVENT = new QName(NS_TYPES, "input-event");
  private static final QName ELEM_OUTPUT_EVENT = new QName(NS_TYPES, "output-event");

  private static final QName ELEM_MESSAGE = new QName(NS_TYPES, "message");
  private static final QName ELEM_PAYLOAD = new QName(NS_TYPES, "payload");
  private static final String ATTR_TYPE = "type";
  private static final QName ELEM_ATTRIBUTES = new QName(NS_TYPES, "attributes");
  private static final QName ELEM_VARIABLES = new QName(NS_TYPES, "variables");
  private static final QName ELEM_VARIABLE = new QName(NS_TYPES, "variable");
  private static final String ATTR_NAME = "name";

  private static final String PARAMETER_NAME = "input";

  @Override
  public Optional<MessageProcessorTypeDeclaration> load(Element element,
                                                        TypeDeclarationLoaderContext typeDeclarationLoaderContext) {
    return XmlMatcher.match(element, ELEM_MP_DECLARATION).map(opDeclaration -> {
      final BaseTypeBuilder typeBuilder = TypeUtils.getTypeBuilder(MetadataFormat.JAVA);
      final FunctionTypeBuilder functionTypeBuilder = typeBuilder.functionType();

      opDeclaration.match(ELEM_INPUT_EVENT).ifPresent(inputEvent -> {
        MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = new MuleEventMetadataTypeBuilder();
        functionTypeBuilder.addParameterOf(PARAMETER_NAME, muleEventMetadataTypeBuilder);
        loadEventMetadataType(muleEventMetadataTypeBuilder, inputEvent, typeDeclarationLoaderContext);
      });
      opDeclaration.match(ELEM_OUTPUT_EVENT).ifPresent(outputEvent -> {
        MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = new MuleEventMetadataTypeBuilder();
        functionTypeBuilder.returnType(muleEventMetadataTypeBuilder);
        loadEventMetadataType(muleEventMetadataTypeBuilder, outputEvent, typeDeclarationLoaderContext);
      });

      return new MessageProcessorTypeDeclaration(functionTypeBuilder.build());
    });
  }

  private void loadEventMetadataType(MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder,
                                     XmlMatcher eventComponentModelMatcher,
                                     TypeDeclarationLoaderContext typeDeclarationLoaderContext) {
    MessageMetadataTypeBuilder messageMetadataTypeBuilder = new MessageMetadataTypeBuilder();
    muleEventMetadataTypeBuilder.message(messageMetadataTypeBuilder);
    eventComponentModelMatcher.match(ELEM_MESSAGE).ifPresent(messageMatcher -> {
      messageMatcher.match(ELEM_PAYLOAD).ifPresent(payloadMatcher -> {
        final String typeExpression = payloadMatcher.requireAttribute(ATTR_TYPE);
        try {
          messageMetadataTypeBuilder
              .payload(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          final Optional<Notifier> notifierOptional = typeDeclarationLoaderContext.getNotifier();
          if (notifierOptional.isPresent()) {
            notifierOptional.get()
                .reportError(findComponentLocation(typeDeclarationLoaderContext),
                             MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("message payload", typeExpression,
                                                                   typeDeclarationLoaderContext.getComponentModel(),
                                                                   e));
          } else {
            Throwables.propagate(e);
          }
        }
      });
      messageMatcher.match(ELEM_ATTRIBUTES).ifPresent(attributesMatcher -> {
        final String typeExpression = attributesMatcher.requireAttribute(ATTR_TYPE);
        try {
          messageMetadataTypeBuilder
              .attributes(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          final Optional<Notifier> notifierOptional = typeDeclarationLoaderContext.getNotifier();
          if (notifierOptional.isPresent()) {
            notifierOptional.get()
                .reportError(findComponentLocation(typeDeclarationLoaderContext),
                             MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("message attributes", typeExpression,
                                                                   typeDeclarationLoaderContext.getComponentModel(),
                                                                   e));
          } else {
            Throwables.propagate(e);
          }
        }
      });
    });
    eventComponentModelMatcher.match(ELEM_VARIABLES).ifPresent(variablesMatcher -> {
      variablesMatcher.matchMany(ELEM_VARIABLE).forEach(variableMatcher -> {
        final String name = variableMatcher.requireAttribute(ATTR_NAME);
        final String typeExpression = variableMatcher.requireAttribute(ATTR_TYPE);
        try {
          muleEventMetadataTypeBuilder
              .addVariable(name, resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          final Optional<Notifier> notifierOptional = typeDeclarationLoaderContext.getNotifier();
          if (notifierOptional.isPresent()) {
            notifierOptional.get()
                .reportError(findComponentLocation(typeDeclarationLoaderContext),
                             MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("variable " + name, typeExpression,
                                                                   typeDeclarationLoaderContext.getComponentModel(),
                                                                   e));
          } else {
            Throwables.propagate(e);
          }
        }
      });
    });
  }
}
