package org.mule.datasense.declarations.loader.xml;

import org.mule.datasense.catalog.model.resolver.TypeResolverException;
import org.mule.datasense.common.loader.xml.XmlMatcher;
import org.mule.datasense.common.util.Notifier;
import org.mule.datasense.common.util.TypeUtils;
import org.mule.datasense.declarations.loader.AbstractTypeDeclarationLoader;
import org.mule.datasense.declarations.loader.TypeDeclarationLoaderContext;
import org.mule.datasense.declarations.model.ExtensionOperationTypeDeclaration;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.FunctionTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;

import com.google.common.base.Throwables;

import java.util.Optional;

import javax.xml.namespace.QName;

import org.w3c.dom.Element;

import static org.mule.datasense.declarations.util.NotificationMessages.MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION;

public class ExtensionOperationTypeDeclarationXmlLoader
    extends AbstractTypeDeclarationLoader<ExtensionOperationTypeDeclaration, Element> {

  private static final String NS_MULE = "http://www.mulesoft.org/schema/mule/types";
  private static final String NS_TYPES = "http://www.mulesoft.org/schema/mule/types";
  private static final QName ELEM_MULE = new QName(NS_TYPES, "mule");
  private static final QName ELEM_OP_DECLARATION = new QName(NS_TYPES, "operation-declaration");
  private static final QName ELEM_INPUTS = new QName(NS_TYPES, "inputs");
  private static final QName ELEM_OUTPUTS = new QName(NS_TYPES, "outputs");
  private static final QName ELEM_PAYLOAD = new QName(NS_TYPES, "payload");
  private static final QName ELEM_ATTRIBUTES = new QName(NS_TYPES, "attributes");
  private static final QName ELEM_PARAMETER = new QName(NS_TYPES, "parameter");
  private static final String ATTR_NAME = "name";
  private static final String ATTR_TYPE = "type";

  @Override
  public Optional<ExtensionOperationTypeDeclaration> load(Element element,
                                                          TypeDeclarationLoaderContext typeDeclarationLoaderContext) {
    return XmlMatcher.match(element, ELEM_OP_DECLARATION).map(opDeclaration -> {
      final BaseTypeBuilder typeBuilder = TypeUtils.getTypeBuilder(MetadataFormat.JAVA);
      final FunctionTypeBuilder functionTypeBuilder = typeBuilder.functionType();
      final Optional<Notifier> notifierOptional = typeDeclarationLoaderContext.getNotifier();

      opDeclaration.match(ELEM_INPUTS).ifPresent(inputs -> {
        inputs.matchMany(ELEM_PARAMETER).forEach(parameterMatcher -> {
          final String name = parameterMatcher.requireAttribute(ATTR_NAME);
          final String typeExpression = parameterMatcher.requireAttribute(ATTR_TYPE);
          try {
            functionTypeBuilder
                .addParameterOf(name, resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
          } catch (TypeResolverException e) {
            if (notifierOptional.isPresent()) {
              notifierOptional.get()
                  .reportError(findComponentLocation(typeDeclarationLoaderContext),
                               MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("parameter " + name, typeExpression,
                                                                     typeDeclarationLoaderContext.getComponentModel(),
                                                                     e));
            } else {
              Throwables.propagate(e);
            }
          }
        });
      });
      opDeclaration.match(ELEM_OUTPUTS).ifPresent(outputs -> {
        MessageMetadataTypeBuilder messageMetadataTypeBuilder = new MessageMetadataTypeBuilder();
        outputs.match(ELEM_PAYLOAD).ifPresent(payloadMatcher -> {
          final String typeExpression = payloadMatcher.requireAttribute(ATTR_TYPE);
          try {
            messageMetadataTypeBuilder
                .payload(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
          } catch (TypeResolverException e) {
            if (notifierOptional.isPresent()) {
              notifierOptional.get()
                  .reportError(findComponentLocation(typeDeclarationLoaderContext),
                               MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("output payload", typeExpression,
                                                                     typeDeclarationLoaderContext.getComponentModel(),
                                                                     e));
            } else {
              Throwables.propagate(e);
            }
          }
        });
        outputs.match(ELEM_ATTRIBUTES).ifPresent(attributesMatcher -> {
          final String typeExpression = attributesMatcher.requireAttribute(ATTR_TYPE);
          try {
            messageMetadataTypeBuilder
                .attributes(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
          } catch (TypeResolverException e) {
            if (notifierOptional.isPresent()) {
              notifierOptional.get()
                  .reportError(findComponentLocation(typeDeclarationLoaderContext),
                               MSG_FAILED_TO_RESOLVE_TYPE_EXPRESSION("output attributes", typeExpression,
                                                                     typeDeclarationLoaderContext.getComponentModel(),
                                                                     e));
            } else {
              Throwables.propagate(e);
            }
          }
        });
        functionTypeBuilder.returnType(messageMetadataTypeBuilder);
      });

      return new ExtensionOperationTypeDeclaration(functionTypeBuilder.build());
    });
  }
}
