package org.mule.datasense.declarations.loader.json;

import org.mule.datasense.catalog.model.resolver.TypeResolverException;
import org.mule.datasense.common.loader.json.JsonMatcher;
import org.mule.datasense.common.util.TypeUtils;
import org.mule.datasense.declarations.loader.AbstractTypeDeclarationLoader;
import org.mule.datasense.declarations.loader.TypeDeclarationLoaderContext;
import org.mule.datasense.declarations.model.MessageProcessorTypeDeclaration;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.FunctionTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;

import com.google.common.base.Throwables;
import com.google.gson.JsonElement;

import java.util.Optional;

public class MessageProcessorTypeDeclarationJsonLoader
    extends AbstractTypeDeclarationLoader<MessageProcessorTypeDeclaration, JsonElement> {

  private static final String PARAMETER_NAME = "input";

  @Override
  public Optional<MessageProcessorTypeDeclaration> load(JsonElement jsonElement,
                                                        TypeDeclarationLoaderContext typeDeclarationLoaderContext) {
    return JsonMatcher.match(jsonElement, "mp_declaration").map(declaration -> {
      final BaseTypeBuilder typeBuilder = TypeUtils.getTypeBuilder(MetadataFormat.JAVA);
      final FunctionTypeBuilder functionTypeBuilder = typeBuilder.functionType();
      declaration.match("input-event").ifPresent(inputEvent -> {
        MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = new MuleEventMetadataTypeBuilder();
        functionTypeBuilder.addParameterOf(PARAMETER_NAME, muleEventMetadataTypeBuilder);
        loadEventMetadataType(muleEventMetadataTypeBuilder, inputEvent, typeDeclarationLoaderContext);
      });
      declaration.match("output-event").ifPresent(outputEvent -> {
        MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = new MuleEventMetadataTypeBuilder();
        functionTypeBuilder.returnType(muleEventMetadataTypeBuilder);
        loadEventMetadataType(muleEventMetadataTypeBuilder, outputEvent, typeDeclarationLoaderContext);
      });
      return new MessageProcessorTypeDeclaration(functionTypeBuilder.build());
    });
  }

  private void loadEventMetadataType(MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder,
                                     JsonMatcher event,
                                     TypeDeclarationLoaderContext typeDeclarationLoaderContext) {

    MessageMetadataTypeBuilder messageMetadataTypeBuilder = new MessageMetadataTypeBuilder();
    muleEventMetadataTypeBuilder.message(messageMetadataTypeBuilder);
    event.match("message").ifPresent(message -> {
      message.match("payload").ifPresent(payload -> {
        final String typeExpression = payload.requireString("type");
        try {
          messageMetadataTypeBuilder
              .payload(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          Throwables.propagate(e);
        }

      });
      message.match("attributes").ifPresent(attributes -> {
        final String typeExpression = attributes.requireString("type");
        try {
          messageMetadataTypeBuilder
              .attributes(resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          Throwables.propagate(e);
        }
      });
    });
    event.matchMany("variables").ifPresent(variables -> {
      variables.forEach(variable -> {
        final String name = variable.requireString("name");
        final String typeExpression = variable.requireString("type");
        try {
          muleEventMetadataTypeBuilder
              .addVariable(name, resolveType(typeExpression, typeDeclarationLoaderContext.getTypesCatalog()));
        } catch (TypeResolverException e) {
          Throwables.propagate(e);
        }
      });
    });
  }
}
