/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stax.StAXSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.soap.SOAPBinding;

import org.apache.commons.lang.StringUtils;
import org.apache.cxf.staxutils.StaxSource;
import org.mule.modules.wsdl.header.SOAPHeaderBuilder;
import org.mule.modules.wsdl.header.SOAPHeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.xml.StaxUtils;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Helper class that invokes SOAP web services.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class WSDLInvocation {

	private static final Logger logger = LoggerFactory
			.getLogger(WSDLInvocation.class);

	/** Tag that represents a Null Payload. */
	private static final Object NULL_PAYLOAD_XML_TAG = "<NullPayload/>";

	/** The instance of {@link XMLInputFactory}. */
	private XMLInputFactory inputFactory;

	/**
	 * The base path for the WSDL. Appending (WSDLname)/?wsdl to this should
	 * return the WSDL.
	 */
	private String basePath;

	/** The {@link SOAPHeaderBuilder} implementation. Optional. */
	private SOAPHeaderBuilder soapHeaderBuilder;

	/** The QName prefix to be used when creating the header. */
	private String headerQNamePrefix;

	/** The parameters to be appended to the header. */
	private Map<String, String> headerParams;

	/** The namespace of the WSDL file. */
	private String wsdlNamespace;

	/** The service name. */
	private String serviceName;

	/** The port name. */
	private String portName;

	/**
	 * Gets the namespace of the WSDL file.
	 *
	 * @return the namespace of the WSDL file.
	 */
	public String getWsdlNamespace() {
		return wsdlNamespace;
	}

	/**
	 * Sets the namespace of the WSDL file.
	 *
	 * @param wsdlNamespace
	 *            the namespace to set.
	 */
	public void setWsdlNamespace(String wsdlNamespace) {
		this.wsdlNamespace = wsdlNamespace;
	}

	/**
	 * Sets the service name.
	 *
	 * @param serviceName
	 *            the service name to set.
	 */
	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}

	/**
	 * Sets the port name.
	 *
	 * @param portName
	 *            the port name to set.
	 */
	public void setPortName(String portName) {
		this.portName = portName;
	}

	/**
	 * Invokes a web service.
	 *
	 * @param type
	 *            The type. A {@link String} with the following format:
	 *            <code>{WSDL name}#{Operation name}</code>.
	 * @param input
	 *            A {link {@link XMLStreamReader reader} with the input message
	 *            to send.
	 * @return A {link {@link XMLStreamReader reader} with the output message
	 *         received.
	 */
	public XMLStreamReader invoke(String type, XMLStreamReader input) {

		XMLStreamReader result = null;

		String wsdlAndOperation[] = type.split("#");
		String wsdlName = wsdlAndOperation[0];
		String operationName = wsdlAndOperation[1];

		String headerPrefix = "headerPrefix";
		String bodyPrefix = "bodyPrefix";
		String methodName = StringUtils.lowerCase(operationName);

		try {
			// Qnames for service as defined in wsdl.
			QName serviceQName = new QName(wsdlNamespace, serviceName);

			// QName for Port As defined in wsdl.
			QName portQName = new QName(wsdlNamespace, portName);

			// Endpoint Address
			String endpointAddress = basePath + wsdlName + "/";

			// Create a dynamic Service instance
			Service service = Service.create(serviceQName);

			// Add a port to the Service
			service.addPort(portQName, SOAPBinding.SOAP11HTTP_BINDING,
					endpointAddress);

			// Create a dispatch instance
			Dispatch<SOAPMessage> dispatch = service.createDispatch(portQName,
					SOAPMessage.class, Service.Mode.MESSAGE);

			// Use Dispatch as BindingProvider
			BindingProvider bp = dispatch;

			// Optionally Configure RequestContext to send SOAPAction HTTP
			// Header
			Map<String, Object> rc = bp.getRequestContext();
			rc.put(BindingProvider.SOAPACTION_USE_PROPERTY, Boolean.TRUE);
			rc.put(BindingProvider.SOAPACTION_URI_PROPERTY, methodName);

			// Obtain a preconfigured SAAJ MessageFactory
			MessageFactory factory = ((SOAPBinding) bp.getBinding())
					.getMessageFactory();

			// Create SOAPMessage Request
			SOAPMessage request = factory.createMessage();

			// Get part
			SOAPPart part = request.getSOAPPart();

			// Gets the elements SOAPEnvelope, header and body.
			SOAPEnvelope env = part.getEnvelope();
			env.setEncodingStyle("http://schemas.xmlsoap.org/soap/encoding/");
			env.addNamespaceDeclaration(bodyPrefix, wsdlNamespace + methodName
					+ "/");
			env.addNamespaceDeclaration(headerPrefix, wsdlNamespace);

			// Request Header
			SOAPHeader header = env.getHeader();

			if (soapHeaderBuilder != null) {
				soapHeaderBuilder.build(header, this);
			}

			if (headerParams != null) {
				for (String key : headerParams.keySet()) {
					header.addChildElement(
							new QName(wsdlNamespace, key, headerQNamePrefix))
							.addTextNode(headerParams.get(key));
				}
			}

			// Request Body
			SOAPBody body = env.getBody();

			// Compose the soap:Body payload
			String xml = getOuterXml(input);

			// handle the case of NullPayload - technically this should only
			// happen when the method has no input param
			if (xml.trim().equals(NULL_PAYLOAD_XML_TAG)) {
				xml = StringUtils.join(new Object[] { "<ns0:", operationName,
						" xmlns:ns0=\"",
						wsdlNamespace.substring(0, wsdlNamespace.length() - 1),
						"\"/>" });
			}
			// http://www.coderanch.com/t/484306/Web-Services/java/Attaching-XML-Message-SoapBody

			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory
					.newInstance();
			docBuilderFactory.setNamespaceAware(true);
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
			Document document = docBuilder.parse(new InputSource(
					new StringReader(xml)));
			body.addDocument(document);

			request.saveChanges();

			// Invoke the endpoint synchronously
			SOAPMessage reply = dispatch.invoke(request);

			// If the service responds a 202 status, the payload will be null.
			// Therefore reply is null. See org.apache.cxf.endpoint.ClientImpl,
			// line 609.
			if (reply == null) {
				return null;
			}

			// Read result
			SOAPBodyElement sourceContent = (SOAPBodyElement) reply
					.getSOAPBody().getChildElements().next();

			// Add the namespaces from the response Envelope tag to the result
			// tag - so it can be parsed successfully by DataMapper
			NamedNodeMap envelopeAttributes = reply.getSOAPPart()
					.getChildNodes().item(0).getAttributes();
			for (int i = 0; i < envelopeAttributes.getLength(); i++) {
				sourceContent.addAttribute(env.createName(envelopeAttributes
						.item(i).getNodeName()), envelopeAttributes.item(i)
						.getNodeValue());
			}
			DOMSource source = new DOMSource(sourceContent);

			XMLInputFactory iFactory = getInputFactory();
			XMLStreamReader reader = iFactory.createXMLStreamReader(source);
			result = StaxUtils.getXMLStreamReader(new StAXSource(reader));
		} catch (SOAPException e) {
			// SOAPFaultException (when web service is down, for example).
			logExceptionDuringInvocation(e);
		} catch (TransformerConfigurationException e) {
			logExceptionDuringInvocation(e);
		} catch (TransformerFactoryConfigurationError e) {
			logExceptionDuringInvocation(e);
		} catch (TransformerException e) {
			logExceptionDuringInvocation(e);
		} catch (ParserConfigurationException e) {
			logExceptionDuringInvocation(e);
		} catch (SAXException e) {
			logExceptionDuringInvocation(e);
		} catch (IOException e) {
			logExceptionDuringInvocation(e);
		} catch (XMLStreamException e) {
			logExceptionDuringInvocation(e);
		} catch (SOAPHeaderException e) {
			logExceptionDuringInvocation(e);
		}

		return result;
	}

	/**
	 * @param e
	 */
	private void logExceptionDuringInvocation(Throwable e) {
		logger.warn("Error during web service invocation", e);
	}

	/**
	 * Gets the {@link XMLInputFactory}, or creates a new one if it was not
	 * created before.
	 *
	 * @return A {@link XMLInputFactory} instance.
	 */
	private XMLInputFactory getInputFactory() {
		if (inputFactory == null) {
			inputFactory = createInputFactory();
		}
		return inputFactory;
	}

	/**
	 * Creates a new {@link XMLInputFactory}instance.
	 *
	 * @return The new {@link XMLInputFactory} instance.
	 */
	private XMLInputFactory createInputFactory() {
		return XMLInputFactory.newInstance();
	}

	/**
	 * Reads a {@link XMLStreamReader} and converts it to {@link String}.
	 *
	 * @param xmlr
	 *            A {@link XMLStreamReader}.
	 * @return The XML, as a {@link String}.
	 * @throws TransformerException
	 *             If there is a problem converting the XML to {@link String}.
	 */
	private String getOuterXml(XMLStreamReader xmlr)
			throws TransformerException {

		Transformer transformer = TransformerFactory.newInstance()
				.newTransformer();
		transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");

		StringWriter stringWriter = new StringWriter();
		transformer.transform(new StaxSource(xmlr), new StreamResult(
				stringWriter));

		return stringWriter.toString();
	}

	/**
	 * Gets the QName prefix to be used when creating the header.
	 *
	 * @return the QName prefix.
	 */
	public String getHeaderQNamePrefix() {
		return headerQNamePrefix;
	}

	/**
	 * Sets the QName prefix to be used when creating the header.
	 *
	 * @param headerQNamePrefix
	 *            the QName prefix to set.
	 */
	public void setHeaderQNamePrefix(String headerQNamePrefix) {
		this.headerQNamePrefix = headerQNamePrefix;
	}

	/**
	 * Sets the parameters to be appended to the header.
	 *
	 * @param headerParams
	 *            the header parameters to set.
	 */
	public void setHeaderParams(Map<String, String> headerParams) {
		this.headerParams = headerParams;
	}

	/**
	 * Sets the base path for the WSDL. Appending (WSDLname)/?wsdl to this
	 * should return the WSDL.
	 *
	 * @param basePath
	 *            the base path to set.
	 */
	public void setBasePath(String basePath) {
		this.basePath = basePath;
	}

	/**
	 * Sets the {@link SOAPHeaderBuilder} implementation.
	 *
	 * @param soapHeaderBuilder
	 *            the {@link SOAPHeaderBuilder} implementation.
	 */
	public void setSoapHeaderBuilder(SOAPHeaderBuilder soapHeaderBuilder) {
		this.soapHeaderBuilder = soapHeaderBuilder;
	}
}
