/**
 * (c) 2003-2012 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

/**
 * This file was automatically generated by the Mule Development Kit
 */
package org.mule.modules.ssh.multiplexer;

import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

import net.schmizz.sshj.SSHClient;
import net.schmizz.sshj.common.IOUtils;
import net.schmizz.sshj.common.StreamCopier;
import net.schmizz.sshj.connection.channel.direct.Session;
import net.schmizz.sshj.connection.channel.direct.Session.Command;
import net.schmizz.sshj.connection.channel.direct.Session.Shell;
import net.schmizz.sshj.transport.TransportException;
import net.schmizz.sshj.userauth.UserAuthException;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.mule.DefaultMuleEvent;
import org.mule.DefaultMuleMessage;
import org.mule.MessageExchangePattern;
import org.mule.api.ConnectionException;
import org.mule.api.ConnectionExceptionCode;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.MuleMessage;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Connect;
import org.mule.api.annotations.ConnectionIdentifier;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Disconnect;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.ValidateConnection;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.lifecycle.Start;
import org.mule.api.annotations.param.ConnectionKey;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.context.MuleContextAware;
import org.mule.construct.Flow;

/**
 * Cloud Connector for ssh that is capable to handle multiple session from
 * different users and reusing the session of any given user if he wants to
 * send multiple messages.
 * 
 * Notice that while this connector can handle N sessions for N users, each user can only have
 * 1 active session at a time
 *
 * @author marianogonzalez
 */
@Connector(name="ssh", schemaVersion="1.2.1", friendlyName="SSH",description="SSH connector")
public class SshConnector implements MuleContextAware {
	
	private static final Logger logger = Logger.getLogger(SshConnector.class);
	
	private static final String SSH_CALLBACK_USER = "SSH_CALLBACK_USER";

	private MuleContext muleContext;
	
	/**
	 * message timeout
	 */
	@Configurable
	private Integer timeout;
	
	/**
	 * If not null, a flow with this name will be
	 * fetched from the registry and invoked everytime
	 * data is received from the other end. Keep in mind that SSH
	 * is a full duplex protocol, meaning that you can receive data at any time,
	 * not just as a reply to a message you have sent.
	 */
	@Configurable
	private String callbackFlowName;
	
	/**
	 * if true, the connection will be opened in shell mode, meaning that
	 * the session context will be maintained from the moment it's stablished
	 * until it's closed. If false, then only single commands will be allowed and no
	 * context will be passed from one invocation to the next.
	 */
	@Configurable
	private boolean shellMode = false;
	
	/**
	 * The flow that will receive callback invocations
	 */
	private Flow callbackFlow = null;

	/**
	 * The size of the receiver buffer in bytes. Defaults to 8192
	 * and must be greater or equal to 1
	 */
	@Configurable
	@Optional
	private Integer receiverBufferSize = 8192;
	
	/**
	 * SSH Client class
	 */
	private SSHClient client;
	
	/**
	 * the SSH session
	 */
	private Session session;
	
	/**
	 * The stream from wich we write
	 */
	private OutputStream writerStream;
	
	private OutputStream callbackStream;
	
	private Shell shell;
	
	/**
	 * The username of the active user
	 */
	private String username;

	@Start
    public void init() {
		this.callbackFlow = (Flow) this.muleContext.getRegistry().lookupFlowConstruct(this.callbackFlowName);
		
		if (this.callbackFlow == null) {
			throw new IllegalArgumentException("Could not find callback flow with name " + this.callbackFlowName);
		}
    }
    
    
    /**
     * Starts a connection
     * @param username the username for the login
     * @param password the password for the login
     * @param host the address for the target host
     * @param port TCP port number in which the host is listening
     * @throws ConnectionException if an error occurs connecting
     */
    @Connect
    public void connect(@ConnectionKey String username, @Password String password, String host, Integer port) throws ConnectionException {
    	this.client = new SSHClient();

    	try {
    		this.client.loadKnownHosts();
    	} catch (IOException e) {
    		throw new ConnectionException(ConnectionExceptionCode.UNKNOWN, e.getMessage(), "Error loading known hosts");
    	}
    	
    	
    	if (this.timeout != null) {
    		this.client.setTimeout(this.timeout);
    		this.client.getTransport().setTimeout(this.timeout);
    	}
    	
    	try {
    		this.client.connect(host, port);
    	} catch (IOException e) {
    		throw new ConnectionException(ConnectionExceptionCode.CANNOT_REACH, e.getMessage(), String.format("Could not reach ssh server at %s:%d", host, port));
    	}
    	
    	try {
    		this.client.authPassword(username, password);
    		this.session = this.client.startSession();
    	} catch (UserAuthException e) {
    		throw new ConnectionException(ConnectionExceptionCode.INCORRECT_CREDENTIALS, e.getMessage(), "Could not login");
    	} catch (TransportException e) {
    		this.throwCannotReachException(e);
    	} catch (net.schmizz.sshj.connection.ConnectionException e) {
    		this.throwCannotReachException(e);
    	}
    	
    	this.username = username;
    	
    	if (this.isShellMode()) {
    		try {
    			this.session.allocateDefaultPTY();
    			this.shell = this.session.startShell();
    		} catch (TransportException e) {
    			this.throwCannotReachException(e);
    		} catch (net.schmizz.sshj.connection.ConnectionException e) {
    			this.throwCannotReachException(e);
    		}
    		
    		int bufSize = shell.getLocalMaxPacketSize();
    		this.writerStream = shell.getOutputStream();
    		this.callbackStream = new CallbackOutputStream(this, bufSize);
    		
    		new StreamCopier(shell.getInputStream(), this.callbackStream)
					.bufSize(bufSize)
					.spawn(String.format("SSH reader thread for user %s", this.username));
			
			new StreamCopier(shell.getErrorStream(), this.callbackStream)
					.bufSize(bufSize)
					.spawn(String.format("SSH error thread for user %s", this.username));
    	}
    }

    /**
     * Releases the connection
     */
    @Disconnect
    public void disconnect() {
    	if (shell != null) {
    		try {
    			shell.close();
    		} catch (Exception e) {
    			logger.error(String.format("Error found closing shell for user %s", this.username), e);
    		}
    	}
    	
    	if (this.session != null) {
    		try {
    			this.session.close();
    		} catch (Exception e) {
    			logger.error(String.format("Error found closing session for user %s", this.username), e);
    		}
    	}
    	
    	if (this.client != null) {
    		try {
    			this.client.disconnect();
    		} catch (IOException e) {
    			logger.error(String.format("Error found closing client for user %s", this.username), e);
    		}
    	}
    }
    
    /**
     * Returns true if the connection is active
     */
    @ValidateConnection
    public boolean isConnected() {
    	return this.client != null
    			&& this.client.isConnected()
    			&& this.client.isAuthenticated()
    			&& this.session != null
    			&& this.session.isOpen()
    			&& (this.shell != null ? this.shell.isOpen() : true);
    }
    
    /**
     * @return the username of the current connection
     */
    @ConnectionIdentifier
    public String getConnectionIdentifier() {
    	return this.username;
    }

    /**
     * Sends the provided content to the SSH host.
     * When the response is received, a new mule message will be generated
     * and will be sent to the configured callback flow.
     * 
     *  That message will carry the response on the payload and will also include
     *  an inbound property called SSH_CALLBACK_USER that specifies the name of the user for whom
     *  the resopnse was sent.
     *  
     * {@sample.xml ../../../doc/ssh-connector.xml.sample ssh:send}
     * 
     * @param content the content to send
     * @param breakLine if true, then a new line character is added at the end of the conent
     * @throws IOException is there was an error in the line
     */
    @Processor
    public void send(String content, @Optional @Default("false") boolean breakLine) throws IOException {
    	content = breakLine ? content + "\n" : content;
    	
    	if (this.shellMode) {
    		this.writerStream.write(content.getBytes());
    		this.writerStream.flush();
    	} else {
    		Command cmd = null;
    		try {
    			cmd = this.session.exec(content);
    			this.doCallback(IOUtils.readFully(cmd.getInputStream()).toString());
    		} finally {
    			try {
    				if (cmd != null) {
    					cmd.close();
    				}
    			} catch (Exception e) {
    				logger.error(String.format("Error found closing command %s for user %s", content, this.username), e);
    			}
    		}
    	}
    }
    
    /**
     * sends the message to the responseFlow if not null
     * @param message - the message. If null then this method does nothing
     */
    protected void doCallback(String response) {
    	if (!StringUtils.isEmpty(response)) {
    		
    		Map<String, Object> inbound = new HashMap<String, Object>();
    		inbound.put(SSH_CALLBACK_USER, username);
    		
    		MuleMessage message = new DefaultMuleMessage(response, inbound, null, null, this.muleContext);
    		MuleEvent event = new DefaultMuleEvent(message, MessageExchangePattern.REQUEST_RESPONSE, this.callbackFlow);
    		
    		try {
    			this.callbackFlow.process(event);
    		} catch (MuleException e) {
				logger.error("Error invoking callback flow", e);
    			throw new RuntimeException(e);
    		}
    	}
    }
    
	public Integer getTimeout() {
		return timeout;
	}


	public void setTimeout(Integer timeout) {
		this.timeout = timeout;
	}
	
	private void throwCannotReachException(Exception e) throws ConnectionException {
		throw new ConnectionException(ConnectionExceptionCode.CANNOT_REACH, e.getMessage(), e.getMessage());
	}


	/**
	 * @return the shellMode
	 */
	public boolean isShellMode() {
		return shellMode;
	}

	public void setShellMode(boolean shellMode) {
		this.shellMode = shellMode;
	}

	/**
	 * @return the callbackFlowName
	 */
	public String getCallbackFlowName() {
		return callbackFlowName;
	}

	public void setCallbackFlowName(String callbackFlowName) {
		this.callbackFlowName = callbackFlowName;
	}

	public Integer getReceiverBufferSize() {
		return receiverBufferSize;
	}

	public void setReceiverBufferSize(Integer receiverBufferSize) {
		if (receiverBufferSize < 1) {
			throw new IllegalArgumentException("ReceiverBufferSize must be greater or equal than 1");
		}
		this.receiverBufferSize = receiverBufferSize;
	}
	
	@Override
	public void setMuleContext(MuleContext context) {
		this.muleContext = context;
		
	}
	
}
