/*
 * GenericTransportManager.java
 *
 * Created on April 14, 2007, 8:03 PM
 */

package com.trilead.ssh2.transport;

import com.trilead.ssh2.ConnectionInfo;
import com.trilead.ssh2.ConnectionMonitor;
import com.trilead.ssh2.DHGexParameters;
import com.trilead.ssh2.ProxyData;
import com.trilead.ssh2.ServerHostKeyVerifier;
import com.trilead.ssh2.crypto.CryptoWishList;
import com.trilead.ssh2.crypto.cipher.BlockCipher;
import com.trilead.ssh2.crypto.digest.MAC;
import com.trilead.ssh2.log.Logger;
import com.trilead.ssh2.packets.PacketDisconnect;
import com.trilead.ssh2.packets.Packets;
import com.trilead.ssh2.packets.TypesReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.SecureRandom;
import java.util.Vector;

/**
 *
 * @author Juraj Bednar, juraj.bednar@digmia.com
 */
public abstract class GenericTransportManager {
    
    protected static final Logger log = Logger.getLogger(GenericTransportManager.class);
    
    protected final Vector asynchronousQueue = new Vector();
    protected Thread asynchronousThread = null;
    String hostname;
    int port;
    
    public GenericTransportManager(String hostname, int port) {
        this.hostname = hostname;
        this.port = port;
    }
    
    boolean connectionClosed = false;
    Vector connectionMonitors = new Vector();
    Object connectionSemaphore = new Object();
    boolean flagKexOngoing = false;
    KexManager km;
    Vector messageHandlers = new Vector();
    boolean monitorsWereInformed = false;
    Throwable reasonClosedCause = null;
    Thread receiveThread;
    TransportConnection tc;
    
    class AsynchronousWorker extends Thread {
        public void run() {
            while (true) {
                byte[] msg = null;
                synchronized (asynchronousQueue) {
                    if (asynchronousQueue.size()  == 0) {
                        /* After the queue is empty for about 2 seconds, stop this thread */
                        
                        try {
                            asynchronousQueue.wait(2000);
                        } catch (InterruptedException e) {
                            /* OKOK, if somebody interrupts us, then we may die earlier. */
                        }
                        
                        if (asynchronousQueue.size()  == 0) {
                            asynchronousThread = null;
                            return;
                        }
                    }
                    
                    msg = (byte[]) asynchronousQueue.remove(0);
                }
                
                /* The following invocation may throw an IOException.
                 * There is no point in handling it - it simply means
                 * that the connection has a problem and we should stop
                 * sending asynchronously messages. We do not need to signal that
                 * we have exited (asynchronousThread = null): further
                 * messages in the queue cannot be sent by this or any
                 * other thread.
                 * Other threads will sooner or later (when receiving or
                 * sending the next message) get the same IOException and
                 * get to the same conclusion.
                 */
                
                try {
                    sendMessage(msg);
                } catch (IOException e) {
                    return;
                }
            }
        }
    }
    
    
    class HandlerEntry {
        MessageHandler mh;
        int low;
        int high;
    }
    
    
    public void changeRecvCipher(BlockCipher bc, MAC mac) {
        tc.changeRecvCipher(bc, mac);
    }
    
    
    public void changeSendCipher(BlockCipher bc, MAC mac) {
        tc.changeSendCipher(bc, mac);
    }
    
    protected abstract void transportClose() throws IOException;
    protected abstract InputStream getInputStream() throws IOException;
    protected abstract OutputStream getOutputStream() throws IOException;
    protected abstract void establishConnection(ProxyData proxyData, int connectTimeout) throws IOException;
    protected ClientServerHello csh;

    public void initialize(CryptoWishList cwl, ServerHostKeyVerifier verifier, DHGexParameters dhgex,
            int connectTimeout, SecureRandom rnd, ProxyData proxyData) throws IOException {
        /* First, establish the TCP connection to the SSH-2 server */
        
        establishConnection(proxyData, connectTimeout);
        
                /* Parse the server line and say hello - important: this information is later needed for the
                 * key exchange (to stop man-in-the-middle attacks) - that is why we wrap it into an object
                 * for later use.
                 */
        
        csh = new ClientServerHello(getInputStream(), getOutputStream());
        
        tc = new TransportConnection(getInputStream(), getOutputStream(), rnd);
        
        km = new KexManager(this, csh, cwl, hostname, port, verifier, rnd);
        km.initiateKEX(cwl, dhgex);
        
        receiveThread = new Thread(new Runnable() {
            public void run() {
                try {
                    receiveLoop();
                } catch (IOException e) {
                    close(e, false);
                    
                    if (log.isEnabled())
                        log.log(10, "Receive thread: error in receiveLoop: " + e.getMessage());
                }
                
                if (log.isEnabled())
                    log.log(50, "Receive thread: back from receiveLoop");
                
                /* Tell all handlers that it is time to say goodbye */
                
                if (km != null) {
                    try {
                        km.handleMessage(null, 0);
                    } catch (IOException e) {
                    }
                }
                
                for (int i = 0; i < messageHandlers.size(); i++) {
                    HandlerEntry he = (HandlerEntry) messageHandlers.elementAt(i);
                    try {
                        he.mh.handleMessage(null, 0);
                    } catch (Exception ignore) {
                    }
                }
            }
        });
        
        receiveThread.setDaemon(true);
        receiveThread.start();
    }
    
    public void close(Throwable cause, boolean useDisconnectPacket) {
        if (useDisconnectPacket == false) {
                        /* OK, hard shutdown - do not aquire the semaphore,
                         * perhaps somebody is inside (and waits until the remote
                         * side is ready to accept new data). */
            
            try {
                transportClose();
            } catch (IOException ignore) {
            }
            
                        /* OK, whoever tried to send data, should now agree that
                         * there is no point in further waiting =)
                         * It is safe now to aquire the semaphore.
                         */
        }
        
        synchronized (connectionSemaphore) {
            if (connectionClosed == false) {
                if (useDisconnectPacket == true) {
                    try {
                        byte[] msg = new PacketDisconnect(Packets.SSH_DISCONNECT_BY_APPLICATION, cause.getMessage(), "")
                        .getPayload();
                        if (tc != null)
                            tc.sendMessage(msg);
                    } catch (IOException ignore) {
                    }
                    
                    try {
                        transportClose();
                    } catch (IOException ignore) {
                    }
                }
                
                connectionClosed = true;
                reasonClosedCause = cause; /* may be null */
            }
            connectionSemaphore.notifyAll();
        }
        
        /* No check if we need to inform the monitors */
        
        Vector monitors = null;
        
        synchronized (this) {
                        /* Short term lock to protect "connectionMonitors"
                         * and "monitorsWereInformed"
                         * (they may be modified concurrently)
                         */
            
            if (monitorsWereInformed == false) {
                monitorsWereInformed = true;
                monitors = (Vector) connectionMonitors.clone();
            }
        }
        
        if (monitors != null) {
            for (int i = 0; i < monitors.size(); i++) {
                try {
                    ConnectionMonitor cmon = (ConnectionMonitor) monitors.elementAt(i);
                    cmon.connectionLost(reasonClosedCause);
                } catch (Exception ignore) {
                }
            }
        }
    }
    
    
    public void forceKeyExchange(CryptoWishList cwl, DHGexParameters dhgex) throws IOException {
        km.initiateKEX(cwl, dhgex);
    }
    
    
    public ConnectionInfo getConnectionInfo(int kexNumber) throws IOException {
        return km.getOrWaitForConnectionInfo(kexNumber);
    }
    
    
    public int getPacketOverheadEstimate() {
        return tc.getPacketOverheadEstimate();
    }
    
    
    public Throwable getReasonClosedCause() {
        synchronized (connectionSemaphore) {
            return reasonClosedCause;
        }
    }
    
    
    public byte[] getSessionIdentifier() {
        return km.sessionId;
    }
    
    
    public void kexFinished() throws IOException {
        synchronized (connectionSemaphore) {
            flagKexOngoing = false;
            connectionSemaphore.notifyAll();
        }
    }
    
    
    public void receiveLoop() throws IOException {
        byte[] msg = new byte[35000];
        
        while (true) {
            int msglen = tc.receiveMessage(msg, 0, msg.length);
            
            int type = msg[0] & 0xff;
            
            if (type == Packets.SSH_MSG_IGNORE)
                continue;
            
            if (type == Packets.SSH_MSG_DEBUG) {
                if (log.isEnabled()) {
                    TypesReader tr = new TypesReader(msg, 0, msglen);
                    tr.readByte();
                    tr.readBoolean();
                    StringBuffer debugMessageBuffer = new StringBuffer();
                    debugMessageBuffer.append(tr.readString("UTF-8"));
                    
                    for (int i = 0; i < debugMessageBuffer.length(); i++) {
                        char c = debugMessageBuffer.charAt(i);
                        
                        if ((c >= 32) && (c <= 126))
                            continue;
                        debugMessageBuffer.setCharAt(i, '\uFFFD');
                    }
                    
                    log.log(50, "DEBUG Message from remote: '" + debugMessageBuffer.toString() + "'");
                }
                continue;
            }
            
            if (type == Packets.SSH_MSG_UNIMPLEMENTED) {
                throw new IOException("Peer sent UNIMPLEMENTED message, that should not happen.");
            }
            
            if (type == Packets.SSH_MSG_DISCONNECT) {
                TypesReader tr = new TypesReader(msg, 0, msglen);
                tr.readByte();
                int reason_code = tr.readUINT32();
                StringBuffer reasonBuffer = new StringBuffer();
                reasonBuffer.append(tr.readString("UTF-8"));
                
                /*
                 * Do not get fooled by servers that send abnormal long error
                 * messages
                 */
                
                if (reasonBuffer.length() > 255) {
                    reasonBuffer.setLength(255);
                    reasonBuffer.setCharAt(254, '.');
                    reasonBuffer.setCharAt(253, '.');
                    reasonBuffer.setCharAt(252, '.');
                }
                
                /*
                 * Also, check that the server did not send charcaters that may
                 * screw up the receiver -> restrict to reasonable US-ASCII
                 * subset -> "printable characters" (ASCII 32 - 126). Replace
                 * all others with 0xFFFD (UNICODE replacement character).
                 */
                
                for (int i = 0; i < reasonBuffer.length(); i++) {
                    char c = reasonBuffer.charAt(i);
                    
                    if ((c >= 32) && (c <= 126))
                        continue;
                    reasonBuffer.setCharAt(i, '\uFFFD');
                }
                
                throw new IOException("Peer sent DISCONNECT message (reason code " + reason_code + "): "
                        + reasonBuffer.toString());
            }
            
            /*
             * Is it a KEX Packet?
             */
            
            if ((type == Packets.SSH_MSG_KEXINIT) || (type == Packets.SSH_MSG_NEWKEYS) || ((type >= 30) && (type <= 49))) {
                km.handleMessage(msg, msglen);
                continue;
            }
            
            MessageHandler mh = null;
            
            for (int i = 0; i < messageHandlers.size(); i++) {
                HandlerEntry he = (HandlerEntry) messageHandlers.elementAt(i);
                if ((he.low <= type) && (type <= he.high)) {
                    mh = he.mh;
                    break;
                }
            }
            
            if (mh == null)
                throw new IOException("Unexpected SSH message (type " + type + ")");
            
            mh.handleMessage(msg, msglen);
        }
    }
    
    
    public void registerMessageHandler(MessageHandler mh, int low, int high) {
        HandlerEntry he = new HandlerEntry();
        he.mh = mh;
        he.low = low;
        he.high = high;
        
        synchronized (messageHandlers) {
            messageHandlers.addElement(he);
        }
    }
    
    
    public void removeMessageHandler(MessageHandler mh, int low, int high) {
        synchronized (messageHandlers) {
            for (int i = 0; i < messageHandlers.size(); i++) {
                HandlerEntry he = (HandlerEntry) messageHandlers.elementAt(i);
                if ((he.mh == mh) && (he.low == low) && (he.high == high)) {
                    messageHandlers.removeElementAt(i);
                    break;
                }
            }
        }
    }
    
    
    public void sendAsynchronousMessage(byte[] msg) throws IOException {
        synchronized (asynchronousQueue) {
            asynchronousQueue.addElement(msg);
            
            /* This limit should be flexible enough. We need this, otherwise the peer
             * can flood us with global requests (and other stuff where we have to reply
             * with an asynchronous message) and (if the server just sends data and does not
             * read what we send) this will probably put us in a low memory situation
             * (our send queue would grow and grow and...) */
            
            if (asynchronousQueue.size()  > 100)
                throw new IOException("Error: the peer is not consuming our asynchronous replies.");
            
            /* Check if we have an asynchronous sending thread */
            
            if (asynchronousThread == null) {
                asynchronousThread = new AsynchronousWorker();
                asynchronousThread.setDaemon(true);
                asynchronousThread.start();
                
                /* The thread will stop after 2 seconds of inactivity (i.e., empty queue) */
            }
        }
    }
    
    
    public void sendKexMessage(byte[] msg) throws IOException {
        synchronized (connectionSemaphore) {
            if (connectionClosed) {
                throw (IOException) new IOException("Sorry, this connection is closed.").initCause(reasonClosedCause);
            }
            
            flagKexOngoing = true;
            
            try {
                tc.sendMessage(msg);
            } catch (IOException e) {
                close(e, false);
                throw e;
            }
        }
    }
    
    
    public void sendMessage(byte[] msg) throws IOException {
        if (Thread.currentThread()  == receiveThread)
            throw new IOException("Assertion error: sendMessage may never be invoked by the receiver thread!");
        synchronized (connectionSemaphore) {
            while (true) {
                if (connectionClosed) {
                    throw (IOException) new IOException("Sorry, this connection is closed.").initCause(reasonClosedCause);
                }
                
                if (flagKexOngoing == false)
                    break;
                
                try {
                    connectionSemaphore.wait();
                } catch (InterruptedException e) {
                }
            }
            
            try {
                tc.sendMessage(msg);
            } catch (IOException e) {
                close(e, false);
                throw e;
            }
        }
    }
    
    
    public void setConnectionMonitors(Vector monitors) {
        synchronized (this) {
            connectionMonitors = (Vector) monitors.clone();
        }
    }
	
    public ClientServerHello getClientServerHello() {
        return csh;
    }
    
    public abstract void setTcpNoDelay(boolean tcpNoDelay) throws IOException;    
}
