/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.soapkit.metadata;

import java.util.Optional;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.module.soapkit.internal.OperationFaultPair;
import org.mule.module.soapkit.internal.SoapkitConfiguration;
import org.mule.module.soapkit.internal.WsdlConnectionInfo;
import org.mule.module.soapkit.internal.metadata.OperationFaultPairResolver;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataResolvingException;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.Is.is;
import static org.mockito.Mockito.when;

@RunWith(MockitoJUnitRunner.class)
public class OperationFaultPairResolverWithMockTestCase {

  private static final OperationFaultPair INVALID_OPERATION_KEY = operationFaultPair("invalidOperation", "invalidFault");
  private static final OperationFaultPair INVALID_FAULT_KEY = operationFaultPair("ListInventory", "invalidFault");

  private static final String TSHIRT_FAULT_METADATA = "{http://mulesoft.org/tshirt-service}TshirtFault";
  private static final String DEFAULT_FAULT_METADATA = "fault";
  private static final String WSDL_LOCATION = "tshirt.wsdl";
  private static final String SERVICE = "TshirtService";
  private static final String PORT = "TshirtServicePort";
  private static final String OPERATION = "ListInventory";
  private static final String FAULT = "TshirtFault";

  @Mock
  private MetadataContext context;

  @Before
  public void setUp() {
    when(context.getConfig()).thenReturn(Optional.of(soapkitConfiguration()));
  }

  @Test(expected = MetadataResolvingException.class)
  public void keyWithinvalidOperation() throws MetadataResolvingException, ConnectionException {
    faultPairResolver().getInputMetadata(context, INVALID_OPERATION_KEY);
  }

  @Test(expected = MetadataResolvingException.class)
  public void keyWithInvalidFault() throws MetadataResolvingException, ConnectionException {
    faultPairResolver().getInputMetadata(context, INVALID_FAULT_KEY);
  }

  @Test
  public void keyWithValidOperationAndFault() throws MetadataResolvingException, ConnectionException {
    final OperationFaultPair key = operationFaultPair(OPERATION, FAULT);
    final MetadataType metadataType = faultPairResolver().getInputMetadata(context, key);
    assertThat(metadataType, notNullValue());
    final ObjectType type = getAndAssertTypeOf(metadataType);
    assertThat(type.getFields(), hasSize(1));
    final ObjectFieldType field = type.getFields().iterator().next();
    assertThat(field.getValue(), is(instanceOf(DefaultObjectType.class)));
    assertThat(field.getKey().getName().toString(), is(TSHIRT_FAULT_METADATA));
  }

  @Test
  public void keyWithOperationAndDefaultFault() throws MetadataResolvingException, ConnectionException {
    final OperationFaultPair key = operationFaultPair(OPERATION, OperationFaultPairResolver.GENERIC_SF);
    final MetadataType metadataType = faultPairResolver().getInputMetadata(context, key);
    assertThat(metadataType, notNullValue());
    final ObjectType type = getAndAssertTypeOf(metadataType);
    assertThat(type.getFields(), hasSize(1));
    final ObjectFieldType field = type.getFields().iterator().next();
    assertThat(field.getValue(), is(instanceOf(DefaultObjectType.class)));
    assertThat(field.getKey().getName().toString(), is(DEFAULT_FAULT_METADATA));
  }

  private ObjectType getAndAssertTypeOf(final MetadataType outputMetadata) {
    assertThat(outputMetadata, is(instanceOf(ObjectType.class)));
    return (ObjectType) outputMetadata;
  }

  private OperationFaultPairResolver faultPairResolver() {
    return new OperationFaultPairResolver();
  }

  private static OperationFaultPair operationFaultPair(final String operation, final String fault) {
    final OperationFaultPair pair = new OperationFaultPair();
    pair.setOperation(operation);
    pair.setFault(fault);
    return pair;
  }

  private static SoapkitConfiguration soapkitConfiguration() {

    final SoapkitConfiguration config = new SoapkitConfiguration();

    final WsdlConnectionInfo info = new WsdlConnectionInfo();
    info.setWsdlLocation(WSDL_LOCATION);
    info.setService(SERVICE);
    info.setPort(PORT);

    config.setInfo(info);
    return config;
  }
}
